#include "mts_matcher.h"

#include <crypta/ext_fp/matcher/lib/matchers/mts_matcher/mts_response_parser.h>
#include <crypta/ext_fp/matcher/lib/matchers/mts_matcher/timestamp_helpers.h>

#include <library/cpp/json/writer/json.h>
#include <library/cpp/neh/http_common.h>
#include <library/cpp/neh/https.h>

#include <util/generic/guid.h>
#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/string/split.h>

using namespace NCrypta::NExtFp::NMatcher;

TMtsMatcher::TMtsMatcher(const TMtsMatcherConfig& config, const TNehSyncMultiClient& client, TStats& stats, NLog::TLogPtr log)
    : Config(config)
    , Client(client)
    , Stats(stats)
    , Log(log)
{
}

TConnection TMtsMatcher::MakeConnection(const TFpEvent& event) {
    return {
        .Ip = event.GetIp(),
        .Port = event.GetPort(),
        .Timestamp = event.GetUnixtime(),
    };
}

void TMtsMatcher::AddConnection(const TFpEvent& event) {
    Stats.Count->Add("events.incoming.mts.count");
    Connections.emplace_back(MakeConnection(event));
}

TMatches TMtsMatcher::GetMatches() {
    if (Connections.size() == 0) {
        return TMatches();
    }
    const auto& requestId = CreateGuidAsString();

    NNeh::TMessage message(GetApiUrl(), "");

    THttpHeaders headers;
    headers.AddHeader("Authorization", "Bearer " + Config.GetToken());
    TStringStream serializedHeaders;
    headers.OutTo(&serializedHeaders);

    const auto& requestBody = GetRequestBody(Connections);
    Log->info("MTS request {} body:\n{}", requestId, requestBody);

    Y_ENSURE(NNeh::NHttp::MakeFullRequest(message, serializedHeaders.Str(), requestBody, "application/json"), "Failed to build request to MTS API");

    Stats.Count->Add("api.calls.mts.count");
    const auto& resp = MakeRequest(Client, message, TDuration::MilliSeconds(Config.GetApiCallTimeoutMs()), "MTS", requestId, Log);

    return NMtsResponseParser::Parse(resp->Data);
}

TString TMtsMatcher::GetApiUrl() const {
    return (Config.GetUseAuthentication() ? "posts://" : "post://") + Config.GetApiUrl();
}

TString TMtsMatcher::GetRequestBody(const TVector<TConnection>& connections) {
    NJsonWriter::TBuf body;

    body.BeginList();

    for (const auto& connection : connections) {
        body.BeginObject()
            .WriteKey("ip").WriteString(connection.Ip)
            .WriteKey("port").WriteInt(connection.Port)
            .WriteKey("ts").WriteULongLong(ToMtsTimestamp(connection.Timestamp))
            .EndObject();
    }

    body.EndList();

    return body.Str();
}
