#include "rostelecom_matcher.h"

#include <crypta/ext_fp/matcher/lib/matchers/rostelecom_matcher/mc_domain/mc_domain.h>

#include <library/cpp/neh/http_common.h>
#include <library/cpp/neh/https.h>
#include <util/generic/guid.h>
#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/string/split.h>

using namespace NCrypta::NExtFp::NMatcher;

TRostelecomMatcher::TRostelecomMatcher(const TRostelecomMatcherConfig& config, const TNehSyncMultiClient& client, TStats& stats, NLog::TLogPtr log)
    : Config(config)
    , Client(client)
    , Stats(stats)
    , Log(log)
{
}

TConnection TRostelecomMatcher::MakeConnection(const TFpEvent& event) {
    return {
        .Ip = event.GetIp(),
        .Port = event.GetPort(),
        .Timestamp = event.GetUnixtime(),
        .Domain = NMcDomain::GetMcDomainForRostelecom(event.GetDuid()),
    };
}

void TRostelecomMatcher::AddConnection(const TFpEvent& event) {
    auto connection = MakeConnection(event);

    Stats.Count->Add("events.incoming.rostelecom.count");
    Request += TStringBuilder() << connection.Ip << '\t'
                                << connection.Port << '\t'
                                << connection.Timestamp << '\t'
                                << connection.Domain << '\n';
}

TMatches TRostelecomMatcher::GetMatches() {
    if (Request.length() == 0) {
        return TMatches();
    }
    const auto& requestId = CreateGuidAsString();
    Log->info("Rostelecom request {} body:\n{}", requestId, Request);

    NNeh::TMessage message(GetApiUrl(), "");
    Y_ENSURE(NNeh::NHttp::MakeFullRequest(message, "", Request, "text/plain"), "Failed to build request to Rostelecom API");

    Stats.Count->Add("api.calls.rostelecom.count");
    const auto& resp = MakeRequest(Client, message, TDuration::MilliSeconds(Config.GetApiCallTimeoutMs()), "Rostelecom", requestId, Log);

    return ParseResponse(resp->Data);
}

TString TRostelecomMatcher::GetApiUrl() const {
    return "post://" + Config.GetApiUrl();
}

TMatches TRostelecomMatcher::ParseResponse(const TString& response) {
    TStringInput stringInput(response);

    TString line;
    TConnection connection;
    TString extId;
    TMatches matches;

    while (stringInput.ReadLine(line)) {
        Split(TStringBuf(line), '\t', connection.Ip, connection.Port, connection.Timestamp, connection.Domain, extId);
        matches[connection] = TMatchResult{.Status = TMatchResult::EStatus::Found, .ExtId = extId};
    }

    return matches;
}
