#include "parser.h"

#include <crypta/lib/native/ip_filter/ip_filter.h>

#include <yt/yt/core/concurrency/scheduler.h>

#include <util/generic/hash.h>
#include <util/generic/ptr.h>
#include <util/generic/vector.h>
#include <util/stream/str.h>
#include <util/string/cast.h>
#include <util/string/split.h>

using namespace NCrypta::NExtFp;
using namespace NCrypta::NExtFp::NMatcher;

TParser::TParser(const TParserConfig& config, NPQ::TCookieQueue& cookiesToCommit, const TProcessor& processor, TStats& stats)
        : Sampler(config.GetSampler().GetDenominator(), config.GetSampler().GetRest())
        , PrefilteredSourceIds(
              config.GetPrefilteredSourceIds().begin(),
              config.GetPrefilteredSourceIds().end(),
              config.GetPrefilteredSourceIds().size()
          )
        , CookiesToCommit(cookiesToCommit)
        , Processor(processor)
        , Log(NLog::GetLog("parser"))
        , Stats(stats)
{
}

void TParser::Parse(NCrypta::NPQ::TConsumer::TReadResult&& readResult) {
    using TAddress = std::pair<TString, ui32>;
    THashMap<TAddress, TLogParser::TParsedLine> addressToLastEvent;

    for (const auto& item : readResult.Data) {
        for (const auto& line : StringSplitter(item).Split('\n').SkipEmpty()) {
            try {
                auto event = TLogParser::ParseLine(line);
                if (!Sampler.Passes(event.GetYuid())) {
                    continue;
                }
                if (IsProviderBound(event.GetDuid(), event.GetIp(), event.GetSourceId())) {
                    auto address = TAddress(event.GetIp(), event.GetPort());
                    auto iter = addressToLastEvent.find(address);
                    if (iter == addressToLastEvent.end()) {
                        addressToLastEvent.emplace(std::move(address), std::move(event));
                    } else if (event.GetUnixtime() > iter->second.GetUnixtime()) {
                        iter->second = std::move(event);
                    }
                }
            } catch (const yexception& e) {
                Stats.Count->Add("parser.errors.bad_event");
                Log->error("Cannot parse json: {}, {}", e.what(), line.Token());
            } catch (...) {
                Stats.Count->Add("parser.errors.bad_event");
                Log->error("Unknown exception: {}", line.Token());
            }
        }
    }

    if (!addressToLastEvent.empty()) {
        TVector<TParsedLine> events;
        events.reserve(addressToLastEvent.size());
        for (auto& [_, event] : addressToLastEvent) {
            events.emplace_back(std::move(event));
        }

        auto future = Processor.ProcessMatches(std::move(events));

        readResult.ClearData();

        NYT::NConcurrency::WaitFor(future).ThrowOnError();
    }

    CookiesToCommit.Enqueue(readResult.EpochCookie);
}

bool TParser::IsProviderBound(ui64 duid, const TString& ip4, const TString& sourceId) const {
    if (ip4 == "0.0.0.0" || duid == 0) {
        return false;
    }

    return PrefilteredSourceIds.contains(sourceId);
}
