#include "processor.h"

#include <crypta/ext_fp/matcher/lib/matchers/beeline_matcher/beeline_matcher.h>
#include <crypta/ext_fp/matcher/lib/matchers/er_telecom_matcher/er_telecom_matcher.h>
#include <crypta/ext_fp/matcher/lib/matchers/intentai_matcher/intentai_matcher.h>
#include <crypta/ext_fp/matcher/lib/matchers/mts_matcher/mts_matcher.h>
#include <crypta/ext_fp/matcher/lib/matchers/rostelecom_matcher/rostelecom_matcher.h>
#include <crypta/lib/native/ext_fp/constants.h>
#include <crypta/lib/native/proto_serializer/proto_serializer.h>
#include <crypta/lib/native/time/scope_timer.h>
#include <crypta/lib/native/time/sleeper.h>
#include <crypta/lib/proto/ext_fp/match_log_entry.pb.h>

#include <yt/yt/core/actions/future.h>
#include <yt/yt/core/concurrency/scheduler.h>
#include <yt/yt/core/concurrency/thread_pool.h>
#include <yt/yt/core/misc/intrusive_ptr.h>

#include <util/stream/str.h>

#include <utility>

using namespace NCrypta;
using namespace NCrypta::NExtFp;
using namespace NCrypta::NExtFp::NMatcher;
using namespace NYT;

TProcessor::TProcessor(
    const TProcessorConfig& config,
    NPQ::TProducer& extFpMatchLogProducer,
    TStats& stats
)
    : Config(config)
    , MatcherConfigs(Config.GetMatchers())
    , ExtFpMatchLogProducer(extFpMatchLogProducer)
    , Log(NLog::GetLog("processor"))
    , Stats(stats)
    , MatchingThreadPool(NYT::New<NYT::NConcurrency::TThreadPool>(config.GetThreads(), "Matcher"))
    , Client("HttpClient")
{
}

NYT::TFuture<void> TProcessor::ProcessMatches(TEvents&& events) const {
    return BIND([this, events = std::move(events)]() -> void {
        Stats.Percentile->Add("event.count", events.size());

        if (events.empty()) {
            return;
        }

        try {
            const auto& matches = GetMatches(events);
            MergeEventsWithMatches(events, matches);
        } catch (const yexception& e) {
            Stats.Count->Add("matching.errors.unexpected");
            Log->error("Matching Error: {}", e.what());
        }
    })
    .AsyncVia(MatchingThreadPool->GetInvoker())
    .Run();
}

TFuture<TMatches> TProcessor::AsyncMatch(IMatcher& matcher, const TString& providerName) const {
    return BIND([this, &matcher, providerName]() -> auto {
               TScopeTimer scopeTimer(Stats.Percentile, "timing.get_" + providerName + "_matches");
               TMatches result;
               try {
                   result = matcher.GetMatches();
               } catch (const yexception& e) {
                   Stats.Count->Add("matching.errors." + providerName);
                   Log->error("Matching Error: {}", e.what());
               }

               return result;
           })
        .AsyncVia(MatchingThreadPool->GetInvoker())
        .Run();
}

TProcessor::TAllMatches TProcessor::GetMatches(const TEvents& events) const {
    TScopeTimer scopeTimer(Stats.Percentile, "timing.get_matches");

    TBeelineMatcher beelineMatcher(MatcherConfigs.GetBeeline(), Client, Stats, Log);
    TErTelecomMatcher erTelecomMatcher(MatcherConfigs.GetErTelecom(), Client, Stats, Log);
    TIntentaiMatcher intentaiMatcher(MatcherConfigs.GetIntentai(), Client, Stats, Log);
    TMtsMatcher mtsMatcher(MatcherConfigs.GetMts(), Client, Stats, Log);
    TRostelecomMatcher rostelecomMatcher(MatcherConfigs.GetRostelecom(), Client, Stats, Log);

    for (const auto& event : events) {
        const auto& sourceId = event.GetSourceId();

        if (sourceId == NExtFp::BEELINE_SOURCE_ID) {
            AddConnection(event, beelineMatcher);
        } else if (sourceId == NExtFp::ER_TELECOM_SOURCE_ID) {
            AddConnection(event, erTelecomMatcher);
        } else if (sourceId == NExtFp::INTENTAI_SOURCE_ID) {
            AddConnection(event, intentaiMatcher);
        } else if (sourceId == NExtFp::MTS_SOURCE_ID) {
            AddConnection(event, mtsMatcher);
        } else if (sourceId == NExtFp::ROSTELECOM_SOURCE_ID) {
            AddConnection(event, rostelecomMatcher);
        } else {
            Stats.Count->Add("events.unknown_source_id");
        }
    }

    auto beelineResult = AsyncMatch(beelineMatcher, NExtFp::BEELINE_SOURCE_ID);
    auto erTelecomResult = AsyncMatch(erTelecomMatcher, NExtFp::ER_TELECOM_SOURCE_ID);
    auto intentaiResult = AsyncMatch(intentaiMatcher, NExtFp::INTENTAI_SOURCE_ID);
    auto rostelecomResult = AsyncMatch(rostelecomMatcher, NExtFp::ROSTELECOM_SOURCE_ID);
    auto mtsResult = AsyncMatch(mtsMatcher, NExtFp::MTS_SOURCE_ID);

    TAllMatches result{
        .Beeline = NYT::NConcurrency::WaitFor(beelineResult).Value(),
        .ErTelecom = NYT::NConcurrency::WaitFor(erTelecomResult).Value(),
        .Intentai = NYT::NConcurrency::WaitFor(intentaiResult).Value(),
        .Mts = NYT::NConcurrency::WaitFor(mtsResult).Value(),
        .Rostelecom = NYT::NConcurrency::WaitFor(rostelecomResult).Value(),
    };

    Stats.Count->Add("api_call.count");
    return result;
}

void TProcessor::AddConnection(const TFpEvent& event, IMatcher& matcher) const {
    matcher.AddConnection(event);
    Stats.Percentile->Add(
        "events.latency." + event.GetSourceId(), TInstant::Now().Seconds() - event.GetUnixtime()
    );
}

void TProcessor::MergeEventsWithMatches(const TEvents& events, const TAllMatches& matches) const {
    auto eventsMatchedTotal = matches.Beeline.size() + matches.ErTelecom.size() + matches.Intentai.size()
        + matches.Mts.size() + matches.Rostelecom.size();
    Stats.Count->Add("events.matched.total", eventsMatchedTotal);

    for (const auto& event : events) {
        const auto& sourceId = event.GetSourceId();
        TMaybe<TString> extId;
        if (sourceId == NExtFp::BEELINE_SOURCE_ID) {
            extId = FindExtId(TBeelineMatcher::MakeConnection(event), matches.Beeline, sourceId);
        } else if (sourceId == NExtFp::ER_TELECOM_SOURCE_ID) {
            extId = FindExtId(TErTelecomMatcher::MakeConnection(event), matches.ErTelecom, sourceId);
        } else if (sourceId == NExtFp::INTENTAI_SOURCE_ID) {
            extId = FindExtId(TIntentaiMatcher::MakeConnection(event), matches.Intentai, sourceId);
        } else if (sourceId == NExtFp::MTS_SOURCE_ID) {
            extId = FindExtId(TMtsMatcher::MakeConnection(event), matches.Mts, sourceId);
        } else {
            extId = FindExtId(TRostelecomMatcher::MakeConnection(event), matches.Rostelecom, sourceId);
        }

        if (!extId.Defined()) {
            continue;
        }

        TMatchLogEntry log;

        log.SetExtSource(sourceId);
        log.SetExtId(*extId);
        log.SetLogType(event.GetLogType());
        log.SetHitlogid(event.GetHitLogId());
        log.SetWatchid(event.GetWatchId());
        log.SetIp(event.GetIp());
        log.SetPort(event.GetPort());
        log.SetDuid(event.GetDuid());
        log.SetUnixtime(event.GetUnixtime());
        log.SetUserAgent(event.GetUserAgent());
        if (event.HasYuid()) {
            log.SetYuid(event.GetYuid());
        }
        log.SetOriginalDomain(event.GetDomain());
        log.SetRtmrTimestamp(event.GetCurrentTimestamp());

        ExtFpMatchLogProducer.TryEnqueue(NCrypta::NProtoSerializer::ToJson(log));
    }
}

TMaybe<TString> TProcessor::FindExtId(const TConnection& connection, const TMatches& matches, const TString& sourceId) const {
    auto it = matches.find(connection);

    if (it == matches.end()) {
        Stats.Count->Add("events.matched." + sourceId + ".status.not_found");
        return Nothing();
    }

    const auto& matchResult = it->second;

    Stats.Count->Add("events.matched." + sourceId + ".status." + ToString(matchResult.Status));

    if (matchResult.Status != TMatchResult::EStatus::Found) {
        return Nothing();
    }

    return matchResult.ExtId;
}
