# encoding: utf-8

import collections
from datetime import timedelta as dt

import crypta.lib.python.ext_fp.constants as ext_fp_consts
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine import logfeller
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.juggler import (
    consts,
    juggler_check_generator,
)
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.solomon import solomon_check_generator
from crypta.lib.python.spine.yasm import yasm_dashboard_generator


PROJECT = "crypta_ext_fp"
MATCHER_SERVICE = "matcher"

EVENT_LATENCY_THRESHOLD_SECS = collections.defaultdict(lambda: 30)
EVENT_LATENCY_THRESHOLD_SECS[ext_fp_consts.MTS_SOURCE_ID] = 32 * 60
EVENT_LATENCY_THRESHOLD_SECS[ext_fp_consts.ROSTELECOM_SOURCE_ID] = 32 * 60
DISABLED_SOURCE_IDS = [ext_fp_consts.MTS_SOURCE_ID]


def get_registry():
    juggler = juggler_check_generator.CryptaYtCheckGenerator(tags=["crypta-ext-fp"])

    yasm = juggler.add_subregistry(yasm_dashboard_generator.CryptaYtYasmDashboardGenerator("External FP"))

    for stream in [
        "crypta@prod@ext-fp-match-log",
    ]:
        logfeller.add_unprased_records_check(juggler, stream)

    add_matcher_checks(juggler, yasm)

    return juggler


def add_matcher_checks(juggler_check_generator, yasm):
    yasm.add_devops(service="Matcher", prj_tag="crypta-ext-fp-matcher")

    for dc, env, nanny_service in [
        ("sas", environment.PRODUCTION, "prod_crypta_ext_fp_matcher_sas"),
        ("vla", environment.PRODUCTION, "prod_crypta_ext_fp_matcher_vla"),
    ]:
        flap_detector_params = FlapDetectorParams(dt(minutes=5), dt(minutes=10))

        juggler = juggler_check_generator.clone(
            child_group=nanny_service,
            child_group_type=consts.GroupType.nanny,
            escalation=False,
            warn_limit=0,
            crit_limit="50%",
        )

        juggler.icmpping().add_flap_detector(flap_detector_params)

        dc_solomon = solomon_check_generator.DcSolomonCheckGenerator(
            juggler,
            project=PROJECT,
            service=MATCHER_SERVICE,
            cluster=env,
            dc=dc,
        )

        if env == environment.PRODUCTION:
            for source_id in ext_fp_consts.SOURCE_IDS:
                if source_id in DISABLED_SOURCE_IDS:
                    continue
                dc_solomon.get_sensor("processor.events.latency.{}.p95".format(source_id)).create_threshold_check(
                    aggregation=alert_pb2.MAX,
                    predicate=alert_pb2.GT,
                    threshold=EVENT_LATENCY_THRESHOLD_SECS[source_id],
                    period=dt(minutes=2),
                    description="event latency (s): {{ pointValue }}",
                ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)

            dc_solomon.get_sensor("processor.timing.get_matches.p95").create_threshold_check(
                aggregation=alert_pb2.MAX,
                predicate=alert_pb2.GT,
                threshold=30 * 10 ** 6,
                period=dt(minutes=2),
                description="API latency (μs): {{ pointValue }}",
            ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)

            dc_solomon.create_sensor_aggregation_check_by_host(
                service="not_found_or_expired",
                period=dt(minutes=2),
                sensor_glob="processor.events.matched.status.expired|processor.events.matched.status.not_found",
                predicate=alert_pb2.GT,
                threshold=300 * 1000,
                description="Unsuccessful matches : {{expression.total}} > {{expression.threshold}}",
            ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)

            dc_solomon.get_sensor("processor.parser.errors.bad_event").create_threshold_check(
                aggregation=alert_pb2.AVG,
                predicate=alert_pb2.GT,
                threshold=100,
                period=dt(minutes=2),
                description="unparsed lines: {{ pointValue }}",
            ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)

            dc_solomon.get_sensor("processor.matching.errors").create_threshold_check(
                aggregation=alert_pb2.AVG,
                predicate=alert_pb2.GT,
                threshold=100,
                period=dt(minutes=2),
                description="matching errors: {{ pointValue }}",
            ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)

            dc_solomon.get_sensor("processor.events.matched.src_conn_missing").create_threshold_check(
                aggregation=alert_pb2.AVG,
                predicate=alert_pb2.GT,
                threshold=100,
                period=dt(minutes=2),
                description="missing matches for connections: {{ pointValue }}",
            ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)

    solomon = solomon_check_generator.ServiceSolomonCheckGenerator(
        juggler_check_generator.clone(
            escalation=False,
        ),
        project=PROJECT,
        service=MATCHER_SERVICE,
        cluster=environment.PRODUCTION,
    )

    solomon.create_sensor_aggregation_check(
        service="total_events_matched",
        period=dt(minutes=2),
        sensor_glob="processor.events.matched.total",
        predicate=alert_pb2.LT,
        threshold=100,
        description="Total events matched : {{expression.total}} < {{expression.threshold}}",
    ).add_flap_detector(flap_detector_params)

    solomon.create_logbroker_lag_check(
        consumer="crypta/prod/ext-fp/consumer",
        topic="crypta/prod/ext-fp/fp-event-log",
        threshold=100 * 1000,
        period=dt(minutes=5),
    ).add_flap_detector(FlapDetectorParams(
        dt(minutes=15),
        dt(minutes=30),
    ))
