from __future__ import print_function

import crypta.lib.python.bt.conf.conf as conf

from collections import defaultdict
from crypta.graph.soup.config.python import ID_TYPE


class AclRolesMeta(type):

    """ List of available ACL roles """

    _ROLES = {
        "crypta-matching-email",
        "crypta-matching-email-hash",
        "crypta-matching-email-phone",
        "crypta-matching-login",
        "crypta-matching-phone",
        "crypta-matching-phone-hash",
        "crypta-matching-puid",
        "crypta-matching-puid-login",
        "crypta-matching-social",
        "crypta-matching-social-private",
        # special cases roles
        "crypta-matching-basic",
        "crypta-matching-all",
        "yandex",
        "crypta-robots",
        # internal roles
        "crypta-matching-yandex-internal",
    }

    def __new__(meta, name, bases, attrs):
        attrs = {key.upper().replace("-", "_"): key for key in meta._ROLES}
        return type.__new__(meta, name, bases, attrs)


class AclRoles:
    __metaclass__ = AclRolesMeta


class AclConfig:

    NON_PERSONAL = [
        ID_TYPE.YANDEXUID.Name,
        ID_TYPE.ICOOKIE.Name,
        ID_TYPE.XUNIQ_GUID.Name,
        ID_TYPE.DISTR_UI.Name,
        ID_TYPE.DISTR_R1.Name,
        ID_TYPE.MAC.Name,
        ID_TYPE.MM_DEVICE_ID.Name,
        ID_TYPE.GAID.Name,
        ID_TYPE.IDFA.Name,
        ID_TYPE.IFV.Name,
        ID_TYPE.OAID.Name,
        ID_TYPE.UUID.Name,
    ]

    YANDEX_PERSONAL = [
        (ID_TYPE.PUID.Name, [AclRoles.CRYPTA_MATCHING_PUID, AclRoles.CRYPTA_MATCHING_PUID_LOGIN]),
        (ID_TYPE.AUTO_ID.Name, [AclRoles.CRYPTA_MATCHING_PUID, AclRoles.CRYPTA_MATCHING_PUID_LOGIN]),
        (ID_TYPE.KINOPOISK_ID.Name, [AclRoles.CRYPTA_MATCHING_PUID, AclRoles.CRYPTA_MATCHING_PUID_LOGIN]),
        (ID_TYPE.LOGIN.Name, [AclRoles.CRYPTA_MATCHING_LOGIN, AclRoles.CRYPTA_MATCHING_PUID_LOGIN]),
        (ID_TYPE.STAFF.Name, [AclRoles.CRYPTA_MATCHING_YANDEX_INTERNAL]),
    ]

    PERSONAL = [
        (
            ID_TYPE.EMAIL_MD5.Name,
            None,
            [
                AclRoles.CRYPTA_MATCHING_EMAIL,
                AclRoles.CRYPTA_MATCHING_EMAIL_HASH,
                AclRoles.CRYPTA_MATCHING_EMAIL_PHONE,
            ],
        ),
        (
            ID_TYPE.EMAIL_SHA256.Name,
            None,
            [
                AclRoles.CRYPTA_MATCHING_EMAIL,
                AclRoles.CRYPTA_MATCHING_EMAIL_HASH,
                AclRoles.CRYPTA_MATCHING_EMAIL_PHONE,
            ],
        ),
        (
            ID_TYPE.PHONE_MD5.Name,
            None,
            [
                AclRoles.CRYPTA_MATCHING_PHONE,
                AclRoles.CRYPTA_MATCHING_PHONE_HASH,
                AclRoles.CRYPTA_MATCHING_EMAIL_PHONE,
            ],
        ),
        (
            ID_TYPE.PHONE_SHA256.Name,
            None,
            [
                AclRoles.CRYPTA_MATCHING_PHONE,
                AclRoles.CRYPTA_MATCHING_PHONE_HASH,
                AclRoles.CRYPTA_MATCHING_EMAIL_PHONE,
            ],
        ),
        (
            ID_TYPE.EMAIL.Name,
            ID_TYPE.EMAIL_MD5.Name,
            [AclRoles.CRYPTA_MATCHING_EMAIL, AclRoles.CRYPTA_MATCHING_EMAIL_PHONE],
        ),
        (
            ID_TYPE.EMAIL.Name,
            ID_TYPE.EMAIL_SHA256.Name,
            [AclRoles.CRYPTA_MATCHING_EMAIL, AclRoles.CRYPTA_MATCHING_EMAIL_PHONE],
        ),
        (
            ID_TYPE.PHONE.Name,
            ID_TYPE.PHONE_MD5.Name,
            [AclRoles.CRYPTA_MATCHING_PHONE, AclRoles.CRYPTA_MATCHING_EMAIL_PHONE],
        ),
        (
            ID_TYPE.PHONE.Name,
            ID_TYPE.PHONE_SHA256.Name,
            [AclRoles.CRYPTA_MATCHING_PHONE, AclRoles.CRYPTA_MATCHING_EMAIL_PHONE],
        ),
        (ID_TYPE.VK_ID.Name, None, [AclRoles.CRYPTA_MATCHING_SOCIAL, AclRoles.CRYPTA_MATCHING_SOCIAL_PRIVATE]),
        (ID_TYPE.OK_ID.Name, None, [AclRoles.CRYPTA_MATCHING_SOCIAL, AclRoles.CRYPTA_MATCHING_SOCIAL_PRIVATE]),
        (ID_TYPE.FB_ID.Name, None, [AclRoles.CRYPTA_MATCHING_SOCIAL, AclRoles.CRYPTA_MATCHING_SOCIAL_PRIVATE]),
    ]

    EMAIL_PHONE = [
        # wrap with tuple to make same logig for all relations t1 = tbl[0]
        (ID_TYPE.EMAIL.Name,),
        (ID_TYPE.EMAIL_MD5.Name,),
        (ID_TYPE.EMAIL_SHA256.Name,),
        (ID_TYPE.PHONE.Name,),
        (ID_TYPE.PHONE_MD5.Name,),
        (ID_TYPE.PHONE_SHA256.Name,),
    ]

    SOCIAL = [(ID_TYPE.VK_ID.Name,), (ID_TYPE.OK_ID.Name,), (ID_TYPE.FB_ID.Name,)]

    EMAIL_HASH = [(ID_TYPE.EMAIL_MD5.Name,), (ID_TYPE.EMAIL_SHA256.Name,)]

    PHONE_HASH = [(ID_TYPE.PHONE_MD5.Name,), (ID_TYPE.PHONE_SHA256.Name,)]

    EMAIL = [(ID_TYPE.EMAIL.Name,), (ID_TYPE.EMAIL_MD5.Name,), (ID_TYPE.EMAIL_SHA256.Name,)]

    PHONE = [(ID_TYPE.PHONE.Name,), (ID_TYPE.PHONE_MD5.Name,), (ID_TYPE.PHONE_SHA256.Name,)]


class MatchingAcl(object):

    BASIC_ROLE = AclRoles.CRYPTA_MATCHING_BASIC
    ALL_ROLE = AclRoles.CRYPTA_MATCHING_ALL
    YANDEX_ROLE = AclRoles.YANDEX
    WRITE_ROLES = [AclRoles.CRYPTA_ROBOTS]

    def __init__(self):
        self._path_per_role = defaultdict(set)
        self._roles_per_path = defaultdict(set)

        def allow_matching(id_type1, id_type2, role):
            self._path_per_role[role].add((id_type1, id_type2))
            self._roles_per_path[(id_type1, id_type2)].add(role)

        role = self.BASIC_ROLE
        for t1 in AclConfig.NON_PERSONAL:
            for t2 in AclConfig.NON_PERSONAL:
                allow_matching(t1, t2, role)
                allow_matching(t2, t1, role)
            allow_matching(t1, t1, role)
            allow_matching(t1, ID_TYPE.CRYPTA_ID.Name, role)
            allow_matching(ID_TYPE.CRYPTA_ID.Name, t1, role)
            allow_matching(ID_TYPE.CRYPTA_ID.Name, ID_TYPE.CRYPTA_ID1.Name, role)
            allow_matching(ID_TYPE.CRYPTA_ID1.Name, ID_TYPE.CRYPTA_ID.Name, role)

        for yandex_personal_type, roles in AclConfig.YANDEX_PERSONAL:
            for role in roles:
                for t2 in AclConfig.NON_PERSONAL:
                    allow_matching(yandex_personal_type, t2, role)
                    allow_matching(t2, yandex_personal_type, role)
                allow_matching(yandex_personal_type, yandex_personal_type, role)
                allow_matching(yandex_personal_type, ID_TYPE.CRYPTA_ID.Name, role)
                allow_matching(ID_TYPE.CRYPTA_ID.Name, yandex_personal_type, role)

        for orig_personal_type, maybe_hash, roles in AclConfig.PERSONAL:
            for role in roles:
                personal_types = [orig_personal_type, maybe_hash] if maybe_hash else [orig_personal_type]
                for personal_type in personal_types:

                    for t2 in AclConfig.NON_PERSONAL:
                        allow_matching(personal_type, t2, role)
                        allow_matching(t2, personal_type, role)

                    for yandex_personal_type, _ in AclConfig.YANDEX_PERSONAL:
                        allow_matching(personal_type, yandex_personal_type, role)
                        allow_matching(yandex_personal_type, personal_type, role)

                    allow_matching(personal_type, personal_type, role)
                    allow_matching(personal_type, ID_TYPE.CRYPTA_ID.Name, role)
                    allow_matching(ID_TYPE.CRYPTA_ID.Name, personal_type, role)

        def cross_roles(from_, to_, role):
            """ Set acl for cross corespond roles """
            for from_table_conf in from_:
                t1 = from_table_conf[0]
                for to_table_conf in to_:
                    t2 = to_table_conf[0]
                    allow_matching(t1, t2, role)
                    allow_matching(t2, t1, role)
                allow_matching(t1, t1, role)

        cross_roles(AclConfig.YANDEX_PERSONAL, AclConfig.YANDEX_PERSONAL, AclRoles.CRYPTA_MATCHING_PUID_LOGIN)
        cross_roles(AclConfig.SOCIAL, AclConfig.PERSONAL, AclRoles.CRYPTA_MATCHING_SOCIAL_PRIVATE)

        cross_roles(AclConfig.EMAIL_PHONE, AclConfig.EMAIL_PHONE, AclRoles.CRYPTA_MATCHING_EMAIL_PHONE)
        cross_roles(AclConfig.EMAIL, AclConfig.EMAIL, AclRoles.CRYPTA_MATCHING_EMAIL)
        cross_roles(AclConfig.PHONE, AclConfig.PHONE, AclRoles.CRYPTA_MATCHING_PHONE)
        cross_roles(AclConfig.EMAIL_HASH, AclConfig.EMAIL_HASH, AclRoles.CRYPTA_MATCHING_EMAIL_HASH)
        cross_roles(AclConfig.PHONE_HASH, AclConfig.PHONE_HASH, AclRoles.CRYPTA_MATCHING_PHONE_HASH)

    def _robots_write_access(self):
        return {
            "action": "allow",
            "inheritance_mode": "object_and_descendants",
            "permissions": ["read", "write", "remove", "administer"],
            "subjects": self.WRITE_ROLES,
        }

    def _custom_read_access(self, roles):
        return {
            "action": "allow",
            "inheritance_mode": "object_and_descendants",  # for attributes
            "permissions": ["read"],
            "subjects": roles,
        }

    def get_pair_acl(self, id_type1, id_type2):
        roles = self._roles_per_path[(id_type1, id_type2)]
        # admin role by default
        roles.add(self.ALL_ROLE)

        return [self._robots_write_access(), self._custom_read_access(list(roles))]

    def get_basic_acl(self):
        return [self._robots_write_access(), self._custom_read_access([self.BASIC_ROLE, self.ALL_ROLE])]

    def get_strict_acl(self):
        return [self._robots_write_access(), self._custom_read_access([self.ALL_ROLE])]

    def get_directory_acl(self):
        return [
            self._robots_write_access(),
            self._custom_read_access([self.BASIC_ROLE, self.ALL_ROLE, self.YANDEX_ROLE]),
        ]

    def print_roles(self):
        for role, pairs in self._path_per_role.items():
            print(role)
            for (id_type1, id_type2) in sorted(pairs):
                print("\t {id_type1} -> {id_type2}".format(id_type1=id_type1, id_type2=id_type2))
            print()

        for (id_type1, id_type2), roles in sorted(self._roles_per_path.items()):
            print("{id_type1} -> {id_type2}".format(id_type1=id_type1, id_type2=id_type2))
            for role in roles:
                print("\t{role}".format(role=role))


class EmptyAcl(object):
    def get_pair_acl(self, id_type1, id_type2):  # noqa
        return []

    def get_basic_acl(self):
        return []

    def get_strict_acl(self):
        return []

    def get_directory_acl(self):
        return []


def get_acl():
    if conf.environment.mode == "develop":
        return EmptyAcl()
    else:
        return MatchingAcl()
