package ru.yandex.crypta.graph.api.model.graph;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import ru.yandex.crypta.graph.api.utils.ExportableEdge;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph.soup.config.proto.TEdgeType;
import ru.yandex.crypta.graph.soup.config.proto.TEdgeUsage;

public class Edge implements ExportableEdge {

    private String id1;
    private String id1Type;
    private String id2;
    private String id2Type;
    private String sourceType;
    private String logSource;
    private Double weight;
    private List<String> dates;
    private String mergeKey = "";
    private GraphElementStatus status = GraphElementStatus.NOT_CHANGED;
    private EdgeUsage usage;
    private List<String> attributes = new ArrayList<>();

    public Edge(Vertex v1, Vertex v2,
                String sourceType, String logSource,
                Double weight, List<String> dates) {
        this(v1.getIdValue(), v1.getIdType(), v2.getIdValue(), v2.getIdType(),
                sourceType, logSource, weight, dates);
    }

    public Edge(String id1, String id1Type, String id2, String id2Type,
                String sourceType, String logSource,
                Double weight, List<String> dates) {
        this.id1 = id1;
        this.id1Type = id1Type;
        this.id2 = id2;
        this.id2Type = id2Type;
        this.sourceType = sourceType;
        this.logSource = logSource;
        this.weight = weight;
        this.dates = dates;

        usage = getEdgeUsage(id1Type, id2Type, sourceType, logSource);
        if (usage == null) {
            usage = getEdgeUsage(id1, id2, sourceType, logSource); // metagraph
        }

        if (usage == null) {
            usage = new EdgeUsage();
        }
    }

    public Edge(String id1, String id1Type, String id2, String id2Type,
                String sourceType, String logSource,
                Double weight, List<String> dates, String mergeKey) {
        this(id1, id1Type, id2, id2Type, sourceType, logSource, weight, dates);
        this.mergeKey = mergeKey;
    }

    public void setAddedStatus() {
        this.status = GraphElementStatus.ADDED;
    }

    public void setRemovedStatus() {
        this.status = GraphElementStatus.REMOVED;
    }

    public void addAttribute(String attr) {
        this.attributes.add(attr);
    }

    public void setAttributes(List<String> attributes) {
        this.attributes = attributes;
    }

    public List<String> getAttributes() {
        return attributes;
    }

    @Override
    public String getId1() {
        return id1;
    }

    @Override
    public String getId1Type() {
        return id1Type;
    }

    @Override
    public String getId2() {
        return id2;
    }

    @Override
    public String getId2Type() {
        return id2Type;
    }

    public Vertex getVertex1() {
        return new Vertex(id1, id1Type);
    }

    public Vertex getVertex2() {
        return new Vertex(id2, id2Type);
    }

    @Override
    public String getSourceType() {
        return sourceType;
    }

    @Override
    public String getLogSource() {
        return logSource;
    }

    public GraphElementStatus getStatus() {
        return status;
    }

    public EdgeUsage getUsage() {
        return usage;
    }

    @Override
    public String getColor() {
        return status.getColor();
    }

    public Double getWeight() {
        return weight;
    }

    public String getMergeKey() {
        return mergeKey;
    }

    public void setMergeKey(String mergeKey) {
        this.mergeKey = mergeKey;
    }

    public List<String> getDates() {
        return dates;
    }


    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Edge)) {
            return false;
        }
        Edge edge = (Edge) o;
        return Objects.equals(id1, edge.id1) &&
                Objects.equals(id1Type, edge.id1Type) &&
                Objects.equals(id2, edge.id2) &&
                Objects.equals(id2Type, edge.id2Type) &&
                Objects.equals(sourceType, edge.sourceType) &&
                Objects.equals(logSource, edge.logSource);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id1, id1Type, id2, id2Type, sourceType, logSource);
    }

    @Override
    public String toString() {
        return id1Type + "_" + id2Type + "_" + sourceType + "_" + logSource +
                "(" + id1 + " -> " + id2 + ")";
    }

    private EdgeUsage getEdgeUsage(String id1Type, String id2Type, String sourceType, String logSource) {
        TEdgeType et = Soup.CONFIG.tryGetEdgeType(id1Type, id2Type, sourceType, logSource);
        if (et != null) {
            TEdgeUsage eu = Soup.CONFIG.getEdgeUsage(et);
            if (eu != null) {
                return new EdgeUsage(eu);
            }
        }
        return null;
    }
}
