package ru.yandex.crypta.graph.api.service;

import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.Stack;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.clients.utils.JsonUtils;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.graph.GraphComponentWithInfo;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.model.ids.GraphIdInfo;
import ru.yandex.crypta.graph.api.service.settings.YtHumanMatchingGraphSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.graph2.dao.yt.utils.YTreeUtils;
import ru.yandex.crypta.graph2.model.soup.props.Login;
import ru.yandex.crypta.lib.yt.JsonMapper;
import ru.yandex.crypta.lib.yt.JsonMultiMapper;
import ru.yandex.crypta.lib.yt.YsonMapper;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.proto.PublicCryptaId;
import ru.yandex.crypta.proto.PublicGraph;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeDoubleNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.lang.StringUtils;

import static java.util.Map.entry;
import static java.util.stream.Collectors.toList;
import static ru.yandex.crypta.graph.api.model.ids.GraphId.CRYPTA_ID_TYPE;
import static ru.yandex.crypta.graph.api.service.settings.YtHumanMatchingGraphSettings.MATCH_SCOPE_NEIGHBOURS;


public class PublicYtGrapthService implements PublicGraphService {

    private static final Logger LOG = LoggerFactory.getLogger(PublicYtGrapthService.class);

    private final YtService yt;
    private YtHumanMatchingGraphSettings graphSettings;
    private static final InfoParams infoParams = new InfoParams(true, false, false, true);
    private final YPath ytVerticesPath = YPath.simple("//home/crypta/production/state/graph/v2/matching/vertices_no_multi_profile");
    private final List<String> idTypes = Arrays.asList(
        "login",
        "phone",
        "email",
        "yandexuid"
    );

    private final Map<String, Integer> publicTypes = Map.ofEntries(
        entry("login", 1),
        entry("email", 2),
        entry("phone", 3),
        entry("gaid", 4),
        entry("idfa", 5),
        entry("oaid", 6),
        entry("mac", 7),
        entry("uuid", 8),
        entry("yandexuid", 9),
        entry("kp_id", 10),
        entry("ok_id", 11)
    );

    Set<String> defaultPermanentActivityTypes = Stream.of(
        "login",
        "email",
        "phone",
        "kp_id",
        "ok_id"
    ).collect(Collectors.toCollection(HashSet::new));

    Set<String> valueAkaTitle = Set.of("login", "email", "phone", "uuid");
    Set<String> techTypes2LeavesRemove = Set.of("yandexuid", "mac");

    private final Map<String, String> appIdToAppName = Map.ofEntries(
        entry("ru.yandex.searchplugin", "Поиск"),
        entry("com.yandex.browser", "Браузер"),
        entry("ru.yandex.taxi", "Такси"),
        entry("ru.yandex.yandexnavi", "Навигатор"),
        entry("ru.yandex.yandexmaps", "Карты"),
        entry("com.yandex.zen", "Дзен"),
        entry("ru.yandex.disk", "Диск"),
        entry("ru.yandex.ytaxi", "Такси"),
        entry("ru.yandex.mail", "Почта"),
        entry("ru.yandex.mobile.navigator", "Навигатор"),
        entry("ru.yandex.music", "Музыка"),
        entry("ru.yandex.uber", "Такси"),
        entry("ru.yandex.traffic", "Карты"),
        entry("ru.yandex.mobile", "Яндекс"),
        entry("ru.yandex.weatherplugin", "Погода"),
        entry("ru.yandex.mobile.search", "Браузер"),
        entry("ru.yandex.translate", "Переводчик"),
        entry("ru.yandex.metro", "Метро"),
        entry("ru.yandex.mail.notificationserviceextension", "Почта"),
        entry("ru.yandex.mobile.music", "Музыка"),
        entry("ru.yandex.blue.market", "Маркет"),
        entry("ru.yandex.mobile.metro", "Метро"),
        entry("ru.yandex.mobile.translate", "Переводчик"),
        entry("com.yandex.browser.lite", "Браузер"),
        entry("ru.yandex.disk.notificationserviceext", "Диск"),
        entry("com.yandex.toloka.androidapp", "Толока"),
        entry("ru.yandex.mobile.weather-v2", "Погода"),
        entry("com.yandex.lavka", "Лавка"),
        entry("ru.yandex.lavka", "Лавка"),
        entry("ru.yandex.ymarket", "Маркет"),
        entry("ru.yandex.mobile.toloka", "Толока"),
        entry("ru.yandex.telemost", "Телемост"),
        entry("ru.yandex.mobile.keyboard.extension", "Клавиатура"),
        entry("ru.yandex.androidkeyboard", "Клавиатура"),
        entry("ru.yandex.blue.market.BeruNotificationServiceExtension", "Беру"),
        entry("ru.yandex.mobile.NotificationService", "Yandex Notification"),
        entry("ru.yandex.mobile.music.widget-extension", "Музыка"),
        entry("ru.yandex.mobile.WhoCallsActionExtension", "Определитель номера"),
        entry("ru.yandex.mobile.search.whocallsaction", "Определитель номера"),
        entry("ru.yandex.mobile.drive", "Драйв"),
        entry("com.yandex.music.auto", "Музыка"),
        entry("ru.yandex.mobile.gasstations", "Заправки"),
        entry("ru.yandex.rasp", "Электрички"),
        entry("ru.yandex.mobile.QuasarShareActionExtension", "Станция"),
        entry("ru.beru.android", "Беру"),
        entry("com.yandex.widget", "Виджет"),
        entry("com.yandex.bus.driver", "Диспетчерская"),
        entry("ru.yandex.driverapp", "Диспетчерская"),
        entry("ru.yandex.androidkeyboard.next", "Клавиатура"),
        entry("ru.yandex.taxi.cookie", "Такси"),
        entry("com.yandex.maps.mrcpublic", "Народная карта"),
        entry("ru.yandex.telemed.doctor", "Кабинет врача"),
        entry("ru.foodfox.courier.debug.releaseserver", "Служба доставки"),
        entry("ru.yandex.taximeter.beta", "Таксометр"),
        entry("com.yandex.yamb.canary", "Ямб"),
        entry("ru.yandex.bus.tickets", "Автобусы"),
        entry("ru.yandex.searchplugin.beta", "Яндекс"),
        entry("ru.yandex.mobile.avia", "Авиабилеты"),
        entry("ru.yandex.direct", "Директ"),
        entry("com.yandex.mobile.drive", "Драйв"),
        entry("ru.foodfox.client", "Еда"),
        entry("ru.foodfox.vendor", "Еда"),
        entry("ru.yandex.med", "Здоровье"),
        entry("com.yandex.courier", "Курьер"),
        entry("ru.yandex.market", "Маркет"),
        entry("com.yandex.mobile.realty", "Недвижимость"),
        entry("ru.yandex.mobile.ofd", "ОФД"),
        entry("ru.yandex.searchplugin.nightly", "Поиск"),
        entry("ru.yandex.poentryka", "Попутка"),
        entry("ru.yandex.mail.beta", "Почта"),
        entry("com.yandex.mobile.job", "Работа"),
        entry("ru.yandex.radio", "Радио"),
        entry("ru.yandex.subtitles", "Разговор"),
        entry("ru.yandex.yandexbus", "Транспорт"),
        entry("ru.yandex.tracker", "Трекер"),
        entry("ru.yandex.startrek", "Трекер"),
        entry("ru.yandex.fines", "Штрафы"),
        entry("ru.yandex.mobile.appmetrica", "AppMetrica"),
        entry("ru.yandex.test.metrica", "AppMetrica"),
        entry("com.yandex.launcher", "Launcher"),
        entry("com.yandex.launcher.externaltheme.football.worldcup", "Football theme"),
        entry("com.yandex.launcher.externaltheme.ny", "New Year theme"),
        entry("ru.yandex.money", "Money"),
        entry("ru.yandex.taximeter", "Taximeter"),
        entry("com.yandex.yamb", "Yamb"),
        entry("ru.yandex.partners", "Advertising Network"),
        entry("com.yandex.browser.alpha", "Browser"),
        entry("com.yandex.browser.beta", "Browser"),
        entry("com.yandex.browser.broteam", "Browser"),
        entry("ru.yandex.nokiasystemservice", "Яндекс"),
        entry("com.yandex.webviewlite.shell", "WebView Shell"),
        entry("ru.yandex.key", "Key"),
        entry("ru.yandex.afisha", "Афиша"),
        entry("ru.yandex.yandextraffic", "Пробки"),
        entry("ru.yandex.mobile.metrica", "Метрики"),
        entry("ru.yandex.auth.client", "Паспорт"),
        entry("ru.yandex.auth.client.am_release_sl", "Паспорт"),
        entry("ru.yandex.shell", "Shell"),
        entry("ru.yandex.taxi.beta", "Такси"),
        entry("ru.yandex.mobile.metrokit.testapp.dev", "MetroKit"),
        entry("ru.yandex.bus.test", "Автобусы"),
        entry("ru.yandex.fotki21", "Фото"),
        entry("ru.yandex.mobile.toloka.dev", "Толока"),
        entry("ru.yandex.mobile.toloka.sandbox", "Толока"),
        entry("ru.yandex.mobile.socialassistant", "Wonder"),
        entry("ru.yandex.mobile.afisha", "Афиша"),
        entry("ru.yandex.mobile.afisha.todayextension", "Афиша"),
        entry("ru.yandex.mobile.search.ipad", "Браузер"),
        entry("ru.yandex.mobile.avia.YaAviaWidget", "Авиа"),
        entry("ru.yandex.mobile.bus.tickets", "Автобусы"),
        entry("ru.yandex.carsharing", "Сarsharing"),
        entry("ru.yandex.mobile.checkout", "Сheckout"),
        entry("ru.yandex.mobile.city", "City"),
        entry("ru.yandex.ytaxi.cookie", "Такси"),
        entry("ru.yandex.mobile.EmbeddedDecoder", "Decoder"),
        entry("ru.yandex.mobile.dictation", "Dictation"),
        entry("ru.yandex.drive", "Drive"),
        entry("ru.yandex.mobile.events", "Events"),
        entry("ru.yandex.fotki.client", "Фото"),
        entry("ru.yandex.mobile.geotoolbox", "Geotoolbox"),
        entry("ru.yandex.mobile.keyboard", "Клавиатура"),
        entry("ru.yandex.mobile.keyboardapp", "Клавиатура"),
        entry("ru.yandex.mobile.keyboardapp.keyboard", "Клавиатура"),
        entry("ru.yandex.khamovniki", "Хамовники"),
        entry("ru.yandex.mobile.maps", "Карты"),
        entry("ru.yandex.traffic.feedback.dev", "Пробки"),
        entry("com.yandex.maps.PushApp.sandbox", "Карты"),
        entry("ru.yandex.mobile.master", "Master"),
        entry("ru.yandex.mobile.medicine", "Здоровье"),
        entry("ru.yandex.mobile.medicine.doctor", "Здоровье"),
        entry("ru.yandex.messenger", "Messenger"),
        entry("ru.yandex.mobile.metrica.today", "Метрика"),
        entry("ru.yandex.mobile.dev", "Яндекс"),
        entry("ru.yandex.mobile.dev.yandextodaywidget", "Яндекс"),
        entry("ru.yandex.mobile.fenerbahce", "Yandex for Fenerbahce"),
        entry("ru.yandex.mobile.dev.imessageextension", "Яндекс"),
        entry("ru.yandex.mobile.next", "Яндекс"),
        entry("ru.yandex.mobile.yandextodaywidget", "Яндекс"),
        entry("ru.yandex.mobile.money", "Money"),
        entry("ru.yandex.mobile.money.messageextension", "Money"),
        entry("ru.yandex.mobile.money.widgetextension", "Money"),
        entry("ru.yandex.mobile.money.extension.widget", "Money"),
        entry("ru.yandex.mobile.moneytransfers", "Money Transfers"),
        entry("ru.yandex.mobile.money.watchkitapp", "Money"),
        entry("ru.yandex.mobile.music.stickers", "Music Stickers"),
        entry("ru.yandex.mobile.navigator.sandbox", "Навигатор"),
        entry("ru.yandex.mobile.news", "Новости"),
        entry("ru.yandex.mobile.operamini", "Opera Mini"),
        entry("ru.yandex.mobile.parkings", "Парковки"),
        entry("ru.yandex.mobile.parkings.widget", "Парковки"),
        entry("ru.yandex.mobile.pereezd", "Переезд"),
        entry("ru.yandex.mobile.personal-jams", "Personal Jams"),
        entry("ru.yandex.mobile.promenad", "Прогулки"),
        entry("ru.yandex.mobile.rabota", "Работа"),
        entry("ru.yandex.mobile.radio", "Радио"),
        entry("ru.yandex.mobile.radio.watchkitapp", "Радио"),
        entry("ru.yandex.mobile.radio.watchkitapp.watchkitextension", "Радио"),
        entry("ru.yandex.rasp.today", "Расписания"),
        entry("ru.yandex.rasp.watch", "Расписания"),
        entry("ru.yandex.rasp.watch.extension", "Расписания"),
        entry("ru.yandex.mobile.realty", "Недвижимость"),
        entry("ru.yandex.red.market", "Макрет"),
        entry("ru.yandex.mobile.teleapp", "Телефония"),
        entry("ru.yandex.mobile.tracker", "Tracker"),
        entry("ru.yandex.traffic.sandbox", "Пробки"),
        entry("ru.yandex.traffic.searchwidget", "Пробки"),
        entry("ru.yandex.traffic.widget", "Пробки"),
        entry("ru.yandex.mobile.translate.translateactionextension", "Переводчик"),
        entry("ru.yandex.mobile.translate.imessageextension", "Переводчик"),
        entry("ru.yandex.mobile.translate.translateshareextension", "Переводчик"),
        entry("ru.yandex.mobile.translate.watchkitapp.watchkitextension", "Переводчик"),
        entry("ru.yandex.mobile.translate.watchkitapp", "Переводчик"),
        entry("ru.yandex.mobile.translate.watchkitextension", "Переводчик"),
        entry("ru.yandex.mobile.transport", "Транспорт"),
        entry("ru.yandex.mobile.tv", "TV"),
        entry("ru.yandex.mobile.weather", "Погода"),
        entry("ru.yandex.mobile.weather-v2.today", "Погода"),
        entry("ru.yandex.mobile.weather-v2.watch", "Погода"),
        entry("ru.yandex.mobile.weather-v2.watch.extension", "Погода"),
        entry("ru.yandex.mobile.yacapp", "Yacapp"),
        entry("ru.yandex.mobile.yamb", "Yamb"),
        entry("ru.yandex.mobile.yamb.extension-share", "Yamb"),
        entry("ru.yandex.mobile.zen", "Дзен"),
        entry("ru.yandex.mobile.fines", "Штрафы"),
        entry("ru.yandex.mobile.yang.dev", "Yang")
    );

    private final DateFormat defaultDtFormat = new SimpleDateFormat("yyyy-MM-dd");
    private final DateFormat textDtFormat = new SimpleDateFormat("dd.MM.yyyy");
    private final long activeThreshold = 90;

    private boolean isYaAppId(String appId) {
        if (
            Pattern.compile("ru\\.yandex\\..*").matcher(appId).find()
                || Pattern.compile("com\\.yandex\\..*").matcher(appId).find()
                || Pattern.compile("yandex\\.auto\\..*").matcher(appId).find()
                || Pattern.compile("ru\\.kinopoisk\\.yandex\\..*").matcher(appId).find()
        ) {
            return true;
        }
        return false;
    }

    @Inject
    public PublicYtGrapthService(
            YtService yt,
            YtHumanMatchingGraphSettings graphSettings
    ) {
        this.yt = yt;
        this.graphSettings = graphSettings;
    }

    private List<Identifier> filterNormalizedId(String id) {
        List<Identifier> filtereIds = new ArrayList<>();

        idTypes.forEach(idType -> {
            Identifier identifier = new Identifier(idType, id);
            if (identifier.isValid()) {
                filtereIds.add(identifier);
            }
        });

        return filtereIds;
    }

    private List<PublicCryptaId> getCryptaidsViaYt(List<Identifier> filtereIds) {
        List<PublicCryptaId> result = new ArrayList<>();

        for (Identifier id : filtereIds) {
            List<String> key = Arrays.asList(
                    id.getNormalizedValue(),
                    id.getType().toString().toLowerCase()
            );

            YPath exactPath = ytVerticesPath.withExact(YtReadingUtils.exact(key));
            Option<JsonNode> info = yt.readTableJson(exactPath, r -> r).firstO();

            if (info.isPresent()) {
                JsonNode graph = info.first();
                result.add(PublicCryptaId.newBuilder()
                    .setCryptaId(graph.get("cryptaId").asText())
                    .setId(graph.get("id").asText())
                    .setIdType(graph.get("id_type").asText())
                    .build()
                );
            }
        }
        return result;
    }

    @Override
    public List<PublicCryptaId> getCryptaId(String id) {

        List<Identifier> filteredIds = filterNormalizedId(id);

        return getCryptaidsViaYt(filteredIds);
    }


    String cleanUaProfile(List<String> list) {
         String title = String.join(" ", list.stream()
             .filter(s -> !s.isEmpty())
             .map(s -> s.replace("\"", ""))
             .collect(toList()));

        return StringUtils.capitalize(title)
            .replace("Ios", "iOS")
            .replace("ios", "iOS");
    }

    String getAppName(String appId) {
        Set<String> set = Stream.of("ru", "yandex", "com").collect(Collectors.toCollection(HashSet::new));

        if (appIdToAppName.containsKey(appId)) {
            return appIdToAppName.get(appId);
        } else {
            List<String> parsed = Arrays.stream(appId.split("\\."))
                .filter(item -> !set.contains(item))
                .collect(toList());

            return String.join(" ", parsed);
        }
    }

    boolean getDefaultActivity(String idType) {
        return defaultPermanentActivityTypes.contains(idType);
    }

    @Override
    public PublicGraph getGraph(String cryptaId) {
        SearchParams searchParams = new SearchParams(null, "v2_prod_yt", "crypta_id", 1);
        GraphId identifier = new GraphId(new Identifier("crypta_id", cryptaId).getNormalizedValue(), "crypta_id");

        Graph graph = getById(identifier, searchParams, infoParams).orElseThrow(Exceptions::notFound);

        PublicGraph.Builder publicGraph = PublicGraph.newBuilder();

        Map<Integer, PublicGraph.IdsInfo> idsInfoMap = new HashMap<>();
        Date today = DateUtils.truncate(new Date(), Calendar.DAY_OF_MONTH);

        graph.getAllEdges().forEach(
            edge -> {
            publicGraph.addEdges(
                PublicGraph.Edge.newBuilder()
                    .setId1(edge.getId1())
                    .setId1Type(edge.getId1Type())
                    .setId2(edge.getId2())
                    .setId2Type(edge.getId2Type())
            );

            putIdsInfo(idsInfoMap, edge.getId1(), edge.getId1Type());
            putIdsInfo(idsInfoMap, edge.getId2(), edge.getId2Type());
        });

        for (GraphIdInfo idInfo : graph.getIdsInfo()) {
            int key = idInfo.getId().hashCode();
            if (idsInfoMap.containsKey(key)) {
                String idValue = idInfo.getId().getIdValue();
                String idType = idInfo.getId().getIdType();

                String uaProfile = "";
                String appName = "";
                String icon = idsInfoMap.get(key).getIcon();
                String device = "";
                if (idInfo.getInfo().containsKey("ua_profile")) {
                    String[] parts = idInfo.getInfo().get("ua_profile").toString()
                            .replace("\"", "")
                            .split("\\|");
                    device = parts.length > 1 ? parts[1] : "";
                    uaProfile = cleanUaProfile(Arrays.stream(parts).skip(2).collect(toList()));
                }

                if (idType.equals("uuid")) {
                    if (idInfo.getInfo().containsKey("app_id")) {
                        icon = idInfo.getInfo().get("app_id").toString().replace("\"", "");
                        String appId = idInfo.getInfo().get("app_id").toString().replace("\"", "");
                        appName = getAppName(appId);
                    }
                }

                String title;
                String header;
                if (idType.equals("yandexuid") || idType.equals("uuid")) {
                    title = getTitle(appName, idType);
                    header = uaProfile.isEmpty() ? idValue : uaProfile;
                } else {
                    title = getTitle(uaProfile.isEmpty() ? idValue : uaProfile, idType);
                    header = title.toLowerCase().equals(idType) ? idValue : idType;
                }

                PublicGraph.IdsInfo idsInfo = idsInfoMap.get(key);

                String dateText = "";
                boolean isActive = getDefaultActivity(idType);
                if (idInfo.getInfo().containsKey("dates")) {
                    try {
                        ObjectMapper mapper = new ObjectMapper();
                        List<String> dates = Arrays.asList(
                                mapper.readValue(idInfo.getInfo().get("dates").toString(), String[].class));
                        if (!dates.isEmpty()) {
                            String firstDateStr = dates.get(0);
                            String lastDateStr = dates.get(dates.size() - 1);

                            Date firstDate = defaultDtFormat.parse(firstDateStr);

                            Date lastDate = defaultDtFormat.parse(lastDateStr);
                            long diffInMillies = today.getTime() - lastDate.getTime();
                            isActive = TimeUnit.DAYS.convert(diffInMillies, TimeUnit.MILLISECONDS) < activeThreshold;
                            if (isActive) {
                                dateText = "Активен с " + textDtFormat.format(firstDate);
                            } else {
                                dateText = "Последняя активность " + textDtFormat.format(lastDate);
                            }
                        }

                    } catch (ParseException e) {
                        e.printStackTrace();
                    } catch (JsonMappingException e) {
                        e.printStackTrace();
                    } catch (JsonParseException e) {
                        e.printStackTrace();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }

                PublicGraph.Popup popup = idsInfo.getPopup().toBuilder()
                    .setHeader(header)
                    .setText(dateText)
                    .build();

                idsInfo = idsInfo.toBuilder()
                    .setTitle(title)
                    .setIsActive(isActive)
                    .setIcon(icon)
                    .setDevice(device)
                    .setPopup(popup)
                    .build();

                idsInfoMap.replace(key, idsInfo);
            }
        }

        idsInfoMap.values().forEach(i -> publicGraph.addIdsInfo(i));

        return publicGraph.build();
    }

    private String getTitle(String idValue, String idType) {
        if (idType.equals("yandexuid")) {
            return "";
        }

        if (valueAkaTitle.contains(idType) && !idValue.isEmpty()) {
            return idValue;
        } else {
            return StringUtils.capitalize(idType);
        }
    }

    private Integer buildtHashCode(String idValue, String idType) {
        return Objects.hash(idValue, idType);
    }

    private void putIdsInfo(Map<Integer, PublicGraph.IdsInfo> idsInfoMap, String idValue, String idType) {
        int idHash = buildtHashCode(idValue, idType);

        if (!idsInfoMap.containsKey(idHash)) {
            String title = getTitle(idValue, idType);
            PublicGraph.Popup popup = PublicGraph.Popup.newBuilder()
                .setHeader(title.equals(idValue) ? idType : idValue)
                .buildPartial();

            PublicGraph.IdsInfo idsInfo = PublicGraph.IdsInfo.newBuilder()
                .setId(idValue)
                .setIdType(idType)
                .setTitle(title)
                .setIcon(idType)
                .setPopup(popup)
                .buildPartial();

            idsInfoMap.put(idHash, idsInfo);
        }
    }

    public static final YsonMapper<Edge> EDGE_MAPPER = (rec) -> {
        Double survivalWeight = YTreeUtils.<YTreeDoubleNode>getNullableNode(rec, "survivalWeight")
            .map(YTreeDoubleNode::getValue)
            .orElse(null);

        Optional<YTreeNode> maybeDates = YTreeUtils.getNullableNode(rec, "dates");
        List<String> dates = maybeDates.stream()
            .flatMap(y -> y.asList().stream().map(YTreeNode::stringValue))
            .collect(toList());

        List<String> shortDates = new ArrayList<>();
        if (dates.size() > 0) {
            shortDates.add(dates.stream().min(String::compareTo).get());

            if (dates.size() > 1) {
                shortDates.add(dates.stream().max(String::compareTo).get());
            }
        }

        Edge edge = new Edge(
            rec.getString("id1"),
            rec.getString("id1Type"),
            rec.getString("id2"),
            rec.getString("id2Type"),
            rec.getString("sourceType"),
            rec.getString("logSource"),
            survivalWeight,
            shortDates,
            rec.getStringO("merge_key").orElse("")
        );

        Optional<YTreeNode> indeviceNode = YTreeUtils.getNullableNode(rec, "indevice");
        if (indeviceNode.isPresent() && indeviceNode.get().boolValue()) {
            edge.addAttribute("indevice");
        }

        return edge;
    };

    private YsonMapper<String> cryptaIdMapper = (rec) -> rec.getString("cryptaId");

    private YsonMapper<Vertex> vertexMapper = (rec) ->
            new Vertex(
                rec.getString("id"),
                rec.getString("id_type")
            );

    private YsonMapper<List<String>> neighboursMapper = (rec) -> {
        YTreeNode neighbours = rec.getOrThrow("neighbours");

        return neighbours.asList().stream().map(YTreeNode::stringValue).collect(Collectors.toList());
    };

    private JsonMapper<GraphId> graphIdMapper = (rec) ->
            new GraphId(rec.get("id").textValue(), rec.get("id_type").textValue());

    private JsonMultiMapper<GraphIdInfo> idsInfoMapper = (recs) ->
            Cf.wrap(recs).groupBy(graphIdMapper)
                .mapEntries((id, infoRecs) -> {
                    GraphIdInfo graphIdInfo = new GraphIdInfo(id);
                    for (JsonNode infoRec : infoRecs) {
                        Map<String, JsonNode> idProps = JsonUtils.jsonToMap(infoRec, r -> r);
                        graphIdInfo.addInfoRec(idProps);
                    }

                    return graphIdInfo;
                });

    private Optional<Graph> getById(GraphId id, SearchParams searchParams, InfoParams infoParams) {
        if (CRYPTA_ID_TYPE.equals(id.getIdType())) {
            return getByCryptaId(id.getIdValue(), searchParams, infoParams);
        } else {
            return getByUsualId(id, searchParams, infoParams);
        }
    }

    private Optional<Graph> getByUsualId(GraphId id, SearchParams params, InfoParams infoParams) {
        YtHumanMatchingGraphSettings.GraphPaths paths = graphSettings.getPaths(params.getMatchingType());

        YPath pathWithKey = paths.getVerticesPath(id);
        Option<String> maybeCryptaId = yt.readTableYson(pathWithKey, cryptaIdMapper).firstO();

        LOG.info("Fetching id->cryptaId from " + pathWithKey.toString());

        return maybeCryptaId.flatMapO(cryptaId -> Option.wrap(
            getByCryptaId(cryptaId, params, infoParams)
        )).toOptional();
    }


    private Optional<Graph> getByCryptaId(String cryptaId, SearchParams params, InfoParams infoParams) {
        YtHumanMatchingGraphSettings.GraphPaths paths = graphSettings.getPaths(params.getMatchingType());
        if (MATCH_SCOPE_NEIGHBOURS.equals(params.getMatchingScope())) {
            return getByCryptaIdWithNeighbours(cryptaId, infoParams, paths);
        } else {
            return getByCryptaIdSingle(cryptaId, infoParams, paths);
        }
    }

    private Optional<Graph> getByCryptaIdSingle(String cryptaId, InfoParams infoParams, YtHumanMatchingGraphSettings.GraphPaths paths) {
        GraphComponentWithInfo singleComponent = getSingleComponent(cryptaId, paths, infoParams);
        if (singleComponent.isEmpty()) {
            return Optional.empty();
        } else {
            Graph graph = new Graph(singleComponent);
            return Optional.of(graph);
        }
    }

    private Optional<Graph> getByCryptaIdWithNeighbours(String cryptaId, InfoParams infoParams, YtHumanMatchingGraphSettings.GraphPaths paths) {
        YPath neighboursTablePath = paths.getGraphNeighboursPath(cryptaId);

        GraphComponentWithInfo mainComponent = getSingleComponent(cryptaId, paths, infoParams.withNeighbours());

        List<String> neighbourCryptaIds =
            yt.readTableYson(neighboursTablePath, neighboursMapper)
                .firstO()
                .<String>flatten()
                .unique().toList();

        List<CompletableFuture<GraphComponentWithInfo>> neighboursResults = neighbourCryptaIds
            .stream()
            .map(neighbour -> CompletableFuture.supplyAsync(
                    () -> getSingleComponent(neighbour, paths, infoParams.forNeighbourComponent())
            )).collect(toList());

        CompletableFuture.allOf(neighboursResults.toArray(new CompletableFuture[0])).join();
        List<GraphComponentWithInfo> neighbourComponents = neighboursResults
            .stream()
            .map(CompletableFuture::join)
            .filter(c -> !c.isEmpty())
            .collect(toList());

        if (mainComponent.isEmpty()) {
            return Optional.empty();

        } else {
            Graph graph = new Graph(mainComponent);

            for (GraphComponentWithInfo component : neighbourComponents) {
                graph.addComponentWithInfo(component);
            }

            return Optional.of(graph);
        }
    }

    private void fillGraph(
            HashMap<Integer, List<Integer>> graph,
            List<Edge> edges,
            HashMap<Integer, Vertex> vertices) {

        for (Edge edge : edges) {
            Vertex v1 = new Vertex(edge.getId1(), edge.getId1Type());
            vertices.put(v1.hashCode(), v1);

            Vertex v2 = new Vertex(edge.getId2(), edge.getId2Type());
            vertices.put(v2.hashCode(), v2);

            addLink(graph, v1, v2);
            addLink(graph, v2, v1);
        }
    }

    private int getPriority(String type, boolean usePuid) {
        return (usePuid && type.equals("puid")) ? 0 : publicTypes.getOrDefault(type, Integer.MAX_VALUE);
    }

    private int vertixCompare(
            Integer l, Integer r,
            HashMap<Integer, Vertex> vertices,
            HashMap<Integer, List<Integer>> graph,
            boolean usePuid) {

        Integer leftPriority = getPriority(vertices.get(l).getIdType(), usePuid);
        Integer rightPriority = getPriority(vertices.get(r).getIdType(), usePuid);

        Integer priorityDiff = leftPriority - rightPriority;

        if (priorityDiff == 0) {
            // incident difference: the more the better
            return graph.get(r).size() - graph.get(l).size();
        } else {
            return priorityDiff;
        }
    }

    private Integer get_root(
            Map<Integer, Boolean> visited,
            HashMap<Integer, Vertex> vertices,
            HashMap<Integer, List<Integer>> graph) {
        return visited.entrySet().stream()
            .filter(e -> !e.getValue())
            .map(Map.Entry::getKey)
            .min((l, r) -> vertixCompare(l, r, vertices, graph, true))
            .orElseThrow();
    }

    private void dfs(
            HashMap<Integer, List<Integer>> graph,
            HashMap<Integer, Vertex> vertices,
            Map<Integer, Integer> depths
            ) {

        Map<Integer, Boolean> visited = new HashMap<>();
        graph.keySet().forEach(v -> visited.put(v, false));

        Stack<Integer> stack = new Stack<>();

        while (visited.containsValue(false)) {
            Integer root = get_root(visited, vertices, graph);
            stack.push(root);
            depths.put(root, 0);

            while (!stack.empty()) {
                Integer vertex = stack.pop();
                if (!visited.get(vertex)) {
                    visited.put(vertex, true);
                }

                Integer currentDepth = depths.get(vertex) + 1;
                List<Integer> sortedNeighbours = graph.get(vertex).stream()
                        .sorted((l, r) -> vertixCompare(l, r, vertices, graph, true))
                        .collect(toList());
                Collections.reverse(sortedNeighbours);
                for (Integer neighbour : sortedNeighbours) {
                    if (!visited.get(neighbour)) {
                        stack.push(neighbour);
                        depths.put(neighbour, currentDepth);
                    }
                }
            }
        }
    }

    void addIncident(List<Integer> incidents, Integer node) {
        if (!incidents.contains(node)) {
            incidents.add(node);
        }
    }

    private Integer getParent(
            HashMap<Integer, List<Integer>> graph,
            Map<Integer, Integer> depths,
            HashMap<Integer, Vertex> vertices,
            Integer transferFrom
            ) {

        Integer parent = graph.get(transferFrom).stream()
            .filter(n -> depths.get(n) < depths.get(transferFrom))
            .min((l, r) -> vertixCompare(l, r, vertices, graph, false))
            .orElse(transferFrom);

        if (parent.equals(transferFrom)) {
            parent = graph.get(transferFrom).stream()
                .min((l, r) -> vertixCompare(l, r, vertices, graph, false))
                .orElse(transferFrom);
        }

        return parent;
    }

    private void transferNodes(
            HashMap<Integer, List<Integer>> graph,
            Map<Integer, Integer> depths,
            HashMap<Integer, Vertex> vertices,
            Integer transferFrom) {

        Integer transferFromDepth = depths.get(transferFrom);

        Integer parent = getParent(graph, depths, vertices, transferFrom);

        if (parent.equals(transferFrom)) {
            LOG.info("ID " + vertices.get(transferFrom).toString() + " don't have parent node. No nodes to transfer");
            return;
        }

        List<Integer> incidentDown = graph.get(transferFrom).stream()
            .filter(n -> (depths.get(n) > transferFromDepth) && !n.equals(parent)).collect(Collectors.toList());

        if (incidentDown.isEmpty()) {
            LOG.info("ID " + vertices.get(transferFrom).toString() + " don't have child node. No nodes to transfer");
            return;
        }

        List<Integer> newParentIncident = new ArrayList<Integer>(graph.get(parent));
        incidentDown.forEach(n -> addIncident(newParentIncident, n));
        graph.put(parent, newParentIncident);

        incidentDown.forEach(n -> {
            List<Integer> newIncident = new ArrayList<Integer>(graph.get(n));
            addIncident(newIncident, parent);
            graph.put(n, newIncident);
        });
    }

    private void removeNode(HashMap<Integer, List<Integer>> graph, Integer toRemove) {
        graph.get(toRemove).forEach(n -> {
            List<Integer> newIncident = new ArrayList<Integer>(graph.get(n));
            newIncident.remove(toRemove);
            if (newIncident.isEmpty()) {
                graph.remove(n);
            } else {
                graph.put(n, newIncident);
            }
        });
        graph.remove(toRemove);
    }

    private boolean isBadNode(Vertex node, Map<Integer, String> appIdsInfo) {
        String idType = node.getIdType();
        String idValue = node.getIdValue();

        if (!publicTypes.containsKey(idType)) {
            return true;
        }

        if (idType.equals("yandexuid") || idType.equals("uuid")) {
            if (!appIdsInfo.containsKey(node.hashCode())) {
                return true;
            }

            if (idType.equals("uuid") && !isYaAppId(appIdsInfo.get(node.hashCode()))) {
                return true;
            }
        } else if (idType.equals("login") || idType.equals("email")) {
            if (
                Login.isSyntheticLogin(idValue)
                || Login.isPhoneGeneratedLogin(idValue)
                || Login.isSocialGeneratedLogin(idValue)
                || idValue.startsWith("yndx-")
            ) {
                return true;
            }
        }

        return false;
    }

    boolean filterLeafTechdNodes(Edge edge, HashMap<Integer, List<Integer>> graph) {
        Integer hashCode1 = buildtHashCode(edge.getId1(), edge.getId1Type());
        Integer hashCode2 = buildtHashCode(edge.getId2(), edge.getId2Type());

        if (graph.containsKey(hashCode1) && graph.containsKey(hashCode2)) {
            if (techTypes2LeavesRemove.contains(edge.getId1Type())) {
                return graph.get(hashCode1).size() > 1;
            }
            if (techTypes2LeavesRemove.contains(edge.getId2Type())) {
                return graph.get(hashCode2).size() > 1;
            }
            return true;
        }
        return false;
    }

    List<Edge> fillEdgesFromAdjacency(
            HashMap<Integer, List<Integer>> graph,
            HashMap<Integer, Vertex> vertices) {

        Set<Edge> finalEdges = new HashSet<>();
        for (Map.Entry<Integer, List<Integer>> entry : graph.entrySet()) {
            Integer node = entry.getKey();
            entry.getValue().forEach(incident -> {
                Integer node1Index = node <= incident ? node : incident;
                Integer node2Index = node > incident ? node : incident;

                Vertex node1 = vertices.get(node1Index);
                Vertex node2 = vertices.get(node2Index);

                Edge edge = new Edge(
                    node1.getIdValue(),
                    node1.getIdType(),
                    node2.getIdValue(),
                    node2.getIdType(),
                    "",
                    "",
                    0D,
                    new ArrayList<>()
                );
                finalEdges.add(edge);
            });
        }
        return new ArrayList<>(finalEdges);
    }

    List<Edge> findYuidYuidEdges(List<Edge> edges) {
        return edges.stream()
                .filter(edge -> edge.getId1Type().equals("yandexuid") && edge.getId2Type().equals("yandexuid"))
                .collect(toList());
    }

    private List<Edge> shrinkGraph(List<Edge> edges, List<GraphIdInfo> idsInfo) {
        edges = edges.stream()
            .filter(edge -> edge.getUsage().isYandexSafe())
            .collect(Collectors.toList());

        HashMap<Integer, List<Integer>> graph = new HashMap<>();
        HashMap<Integer, Vertex> vertices = new HashMap<>();

        fillGraph(graph, edges, vertices);

        Map<Integer, Integer> depths = new HashMap<>();
        dfs(graph, vertices, depths);

        Map<Integer, String> appIdsInfo = new HashMap<>();
        idsInfo.forEach(
            id -> appIdsInfo.put(
                id.getId().hashCode(),
                id.getInfo().containsKey("app_id") ? id.getInfo().get("app_id").toString().replace("\"", "") : ""
            ));

        List<Integer> sortedVertices = vertices.keySet()
            .stream()
            .sorted((l, r) -> vertixCompare(l, r, vertices, graph, true))
            .collect(toList());
        Collections.reverse(sortedVertices);

        for (Integer key : sortedVertices) {
            Vertex vertex = vertices.get(key);
            if (isBadNode(vertex, appIdsInfo)) {
                try {
                    transferNodes(graph, depths, vertices, key);
                    removeNode(graph, key);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        List<Edge> edgesAfterRemoveBadNodes = fillEdgesFromAdjacency(graph, vertices);

        List<Edge> yuidYuidEdges = findYuidYuidEdges(edgesAfterRemoveBadNodes);

        while (!yuidYuidEdges.isEmpty()) {
            for (Edge edge : yuidYuidEdges) {
                Integer hashCode1 = buildtHashCode(edge.getId1(), edge.getId1Type());
                Integer hashCode2 = buildtHashCode(edge.getId2(), edge.getId2Type());
                if (graph.containsKey(hashCode1) && graph.containsKey(hashCode2)) {
                    transferNodes(graph, depths, vertices, hashCode1);
                    removeNode(graph, hashCode1);
                }
            }
            edgesAfterRemoveBadNodes = fillEdgesFromAdjacency(graph, vertices);
            yuidYuidEdges = findYuidYuidEdges(edgesAfterRemoveBadNodes);
        }

        return edgesAfterRemoveBadNodes.stream()
            .filter(edge -> filterLeafTechdNodes(edge, graph))
            .collect(toList());
    }

    private void addLink(Map<Integer, List<Integer>> graph, Vertex v1, Vertex v2) {
        if (!graph.containsKey(v1.hashCode())) {
            graph.put(v1.hashCode(), Arrays.asList(v2.hashCode()));
        } else {
            List<Integer> neighbors = new ArrayList<>(graph.get(v1.hashCode()));
            if (!neighbors.contains(v2.hashCode())) {
                neighbors.add(v2.hashCode());
                graph.put(v1.hashCode(), neighbors);
            }
        }
    }

    List<String> getRange(List<String> strings) {
        List<String> result = new ArrayList<>();
        if (strings.isEmpty()) {
            return result;
        }

        String minStr = strings.stream().min(String::compareTo).get();
        String maxStr = strings.stream().max(String::compareTo).get();

        result.add(minStr);
        if (!minStr.equals(maxStr)) {
            result.add(maxStr);
        }

        return result;
    }

    void updateDatesFromNode(
            String idValue,
            String idType,
            List<String> dates,
            Map<Integer, Integer> idsIndex,
            List<GraphIdInfo> updatedIdsInfo) {

        Integer hashCode = buildtHashCode(idValue, idType);

        if (idsIndex.containsKey(hashCode)) {
            Integer index = idsIndex.get(hashCode);
            Map<String, JsonNode> idInfo = updatedIdsInfo.get(index).getInfo();
            if (idInfo.containsKey("dates")) {
                try {
                    ObjectMapper mapper = new ObjectMapper();
                    dates.addAll(Arrays.asList(mapper.readValue(idInfo.get("dates").toString(), String[].class)));
                    dates = getRange(dates);

                    String datesStr = mapper.writeValueAsString(dates);
                    JsonNode datesJson = mapper.readTree(datesStr);
                    idInfo.replace("dates", datesJson);

                    GraphIdInfo updatedNode = new GraphIdInfo(
                            updatedIdsInfo.get(index).getId(),
                            idInfo
                    );

                    updatedIdsInfo.set(index, updatedNode);

                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        } else {
            try {
                ObjectMapper mapper = new ObjectMapper();
                String datesStr = mapper.writeValueAsString(dates);
                JsonNode datesJson = mapper.readTree(datesStr);
                Map<String, JsonNode> jsonProps = Map.of(
                    "dates", datesJson
                );

                GraphIdInfo newNode = new GraphIdInfo(
                        new GraphId(idValue, idType),
                        jsonProps
                );
                updatedIdsInfo.add(newNode);
                idsIndex.put(hashCode, updatedIdsInfo.size()-1);

            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    List<GraphIdInfo> updateDatesFromEdges(List<GraphIdInfo> idsInfo, List<Edge> edges) {
        ListIterator<GraphIdInfo> it = idsInfo.listIterator();
        Map<Integer, Integer> idsIndex = new HashMap<>();

        while (it.hasNext()) {
            GraphIdInfo id = it.next();
            idsIndex.put(id.getId().hashCode(), it.nextIndex());
        }

        List<GraphIdInfo> updatedIdsInfo = new ArrayList<>(idsInfo);

        for (Edge edge : edges) {
            List<String> dates = getRange(edge.getDates());

            updateDatesFromNode(
                edge.getId1(),
                edge.getId1Type(),
                dates,
                idsIndex,
                updatedIdsInfo
            );

            updateDatesFromNode(
                edge.getId2(),
                edge.getId2Type(),
                dates,
                idsIndex,
                updatedIdsInfo
            );
        }

        return updatedIdsInfo;
    }

    List<GraphIdInfo> filterIdsByEdges(List<GraphIdInfo> idsInfo, List<Edge> edges) {
        Set<Integer> linkedNodes = new HashSet<>();
        edges.forEach(edge -> {
            linkedNodes.add(buildtHashCode(edge.getId1(), edge.getId1Type()));
            linkedNodes.add(buildtHashCode(edge.getId2(), edge.getId2Type()));
        });

        return idsInfo.stream()
                .filter(graphIdInfo -> linkedNodes.contains(graphIdInfo.getId().hashCode()))
                .collect(toList());
    }

    private GraphComponentWithInfo getSingleComponent(String cryptaId, YtHumanMatchingGraphSettings.GraphPaths paths, InfoParams infoParams) {

        CompletableFuture<List<Edge>> edgesResult = getEdges(cryptaId, paths);
        CompletableFuture<List<GraphIdInfo>> idsInfoResult = getIdsInfo(cryptaId, paths, infoParams);

        CompletableFuture.allOf(
            edgesResult,
            idsInfoResult
        ).join();

        List<GraphIdInfo> idsInfo = idsInfoResult.join();

        List<Edge> edges = edgesResult.join();

        idsInfo = updateDatesFromEdges(idsInfo, edges);

        edges = shrinkGraph(edges, idsInfo);

        idsInfo = filterIdsByEdges(idsInfo, edges);

        List<Vertex> vertices;
        if (edges.isEmpty()) {
            // case when component contains from single vertex
            vertices = yt.readTableYson(paths.getVerticesByCryptaIdPath(cryptaId), vertexMapper);
        } else {
            vertices = new ArrayList<>(GraphComponent.verticesOfEdges(edges));
        }

        GraphComponent graphComponent = new GraphComponent(cryptaId, vertices, edges);
        GraphComponentWithInfo withInfo = new GraphComponentWithInfo(graphComponent);

        withInfo.setIdsInfo(idsInfo);

        return withInfo;
    }

    private CompletableFuture<List<Edge>> getEdges(String cryptaId, YtHumanMatchingGraphSettings.GraphPaths paths) {
        return CompletableFuture.supplyAsync(() -> yt.readTableYson(paths.getEdgesPath(cryptaId), EDGE_MAPPER));
    }

    private CompletableFuture<List<GraphIdInfo>> getIdsInfo(String cryptaId, YtHumanMatchingGraphSettings.GraphPaths paths, InfoParams infoParams) {
        if (infoParams.includeIdsInfo) {
            YPath verticesPropertiesPath = paths.getVerticesPropertiesPath(cryptaId);
            List<JsonNode> idInfoRecs = yt.readTableJson(verticesPropertiesPath, (r) -> r);
            return CompletableFuture.supplyAsync(() -> idsInfoMapper.apply(idInfoRecs));
        } else {
            return CompletableFuture.completedFuture(List.of());
        }
    }
}
