package ru.yandex.crypta.graph.api.service;

import java.util.Optional;

import javax.ws.rs.core.SecurityContext;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.YtAntifraudSoupGraphSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.graph.api.utils.SecureGraphHelper;
import ru.yandex.crypta.idm.Roles;

import static ru.yandex.crypta.graph.api.utils.SecureGraphHelper.isPersonalId;

public class SecureGraphService implements GraphService {

    private final SecurityContext securityContext;
    private final GraphService graphService;

    public SecureGraphService(SecurityContext securityContext, GraphService graphService) {
        this.securityContext = securityContext;
        this.graphService = graphService;
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams params, InfoParams infoParams) {
        boolean userCanWatchPrivate = securityContext.isUserInRole(Roles.Portal.GRAPH_PRIVATE);

        if (isPersonalId(id.getIdType()) && !userCanWatchPrivate) {
            throw Exceptions.forbidden(
                    "You must have a role: Crypta -> Portal -> Graph with personal ids",
                    "IDM_ROLE_MISSED"
            );
        }

        boolean userCanWatchAntifraud = securityContext.isUserInRole(Roles.Portal.GRAPH_ANTIFRAUD);
        if (YtAntifraudSoupGraphSettings.MATCH_TYPES.contains(params.getMatchingType()) && !userCanWatchAntifraud) {
            throw Exceptions.forbidden(
                    "You must have a role: Crypta -> Portal -> Graph with antifraud ids",
                    "IDM_ROLE_MISSED"
            );
        }

        Graph graph = graphService.getById(id, params, infoParams).orElseThrow(Exceptions::notFound);
        if (userCanWatchPrivate) {
            return Optional.of(graph);
        } else {
            SecureGraphHelper secureGraphHelper = new SecureGraphHelper();
            Graph maskedGraph = secureGraphHelper.createMaskedGraph(graph);
            return Optional.of(maskedGraph);
        }
    }

    @Override
    public GraphSettings getGraphSettings() {
        return graphService.getGraphSettings();
    }


}
