package ru.yandex.crypta.graph.api.service;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.inject.Inject;

import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.SoupConfigGraphSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;

public class SoupConfigGraphService implements GraphService {

    private static final String ID_TYPE_TYPE = "id_type";
    private final GraphSettings graphSettings;

    @Inject
    public SoupConfigGraphService(SoupConfigGraphSettings graphSettings) {
        this.graphSettings = graphSettings;
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams params, InfoParams infoParams) {


        List<Vertex> vertices = Arrays.stream(EIdType.values())
                .filter(idType -> !idType.equals(EIdType.UNRECOGNIZED) && !idType.equals(EIdType.DEFAULT))
                .map(idType -> new Vertex(
                                Soup.CONFIG.name(idType),
                                ID_TYPE_TYPE
                        )
                ).collect(Collectors.toList());

        List<Edge> edges = Soup.CONFIG.getEdgeTypes().stream()
                .map(edgeType -> {
                            String id1Type = Soup.CONFIG.name(edgeType.getId1Type());
                            String id2Type = Soup.CONFIG.name(edgeType.getId2Type());
                            String sourceType = Soup.CONFIG.name(edgeType.getSourceType());
                            if (sourceType.contains("partner")) {
                                // deduplicate partners
                                sourceType = "partner";
                            }
                            String logSource = Soup.CONFIG.name(edgeType.getLogSource());

                            return new Edge(
                                    id1Type, ID_TYPE_TYPE, id2Type, ID_TYPE_TYPE,
                                    sourceType, logSource,
                                    0.0, Collections.emptyList());
                        }
                ).distinct().collect(Collectors.toList());

        return Optional.of(new Graph(
                new GraphComponent("Meta Soup", vertices, edges)
        ));
    }

    @Override
    public GraphSettings getGraphSettings() {
        return graphSettings;
    }
}
