package ru.yandex.crypta.graph.api.service;

import java.util.Optional;

import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.graph.api.service.transformer.ExtendWithSoupEdgesTransformer;
import ru.yandex.crypta.graph.api.service.transformer.GraphShrinkingTransformer;
import ru.yandex.crypta.graph.api.service.transformer.GraphTransformer;
import ru.yandex.crypta.graph.api.service.transformer.IndeviceGraphTransformer;
import ru.yandex.crypta.graph.api.service.transformer.SpanningActiveTransformer;
import ru.yandex.crypta.graph.api.service.transformer.SpanningTreeTransformer;
import ru.yandex.crypta.lib.yt.YtService;

public class TransformerGraphService implements GraphService {

    public static final String MATCH_SCOPE_SPANNING = "spanning";
    public static final String MATCH_SCOPE_SPANNING_ACTIVE = "spanning_active";
    public static final String MATCH_SCOPE_SPANNING_ACTIVE_2 = "spanning_active_2";
    public static final String MATCH_SCOPE_KEEP_SOUPY_INDEVICE = "keep_indevice_only";
    public static final String MATCH_SCOPE_EXTEND_WITH_INNER_SOUP = "extend_with_inner_soup_edges";
    public static final String MATCH_SCOPE_EXTEND_WITH_SOUP = "extend_with_soup_edges";

    private final GraphService graphService;
    private final YtService ytService;

    public TransformerGraphService(GraphService graphService, YtService ytService) {
        this.graphService = graphService;
        this.ytService = ytService;
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams searchParams, InfoParams infoParams) {
        Optional<Graph> graphO = graphService.getById(id, searchParams, infoParams);
        return graphO.map(graph -> {
            GraphTransformer transformer = getTransformer(graph, searchParams.getMatchingScope());
            if (transformer != null) {
                return transformer.transform(graph);
            } else {
                return graph;
            }
        });
    }


    private GraphTransformer getTransformer(Graph graph, String matchScope) {
        // TODO: spanning active shouldn't know graph during construction
        if (MATCH_SCOPE_SPANNING_ACTIVE.equals(matchScope)) {
            return new SpanningActiveTransformer(graph, false);

        } else if (MATCH_SCOPE_SPANNING_ACTIVE_2.equals(matchScope)) {
            return new SpanningActiveTransformer(graph, true);

        } else if (MATCH_SCOPE_SPANNING.equals(matchScope)) {
            return new SpanningTreeTransformer();

        } else if (GraphShrinkingTransformer.MATCH_SCOPES.contains(matchScope)) {
            var shrinkType = GraphShrinkingTransformer.ShrinkType.valueOf(matchScope);
            return new GraphShrinkingTransformer(shrinkType);
        } else if (MATCH_SCOPE_KEEP_SOUPY_INDEVICE.equals(matchScope)) {
            return new IndeviceGraphTransformer();
        } else if (MATCH_SCOPE_EXTEND_WITH_INNER_SOUP.equals(matchScope)) {
            return new ExtendWithSoupEdgesTransformer(ytService, true);
        } else if (MATCH_SCOPE_EXTEND_WITH_SOUP.equals(matchScope)) {
            return new ExtendWithSoupEdgesTransformer(ytService, false);
        } else {
            return null;
        }
    }

    @Override
    public GraphSettings getGraphSettings() {
        return graphService.getGraphSettings();
    }

}
