package ru.yandex.crypta.graph.api.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.google.protobuf.InvalidProtocolBufferException;
import crypta.idserv.proto.Graph.TAttribute;
import crypta.idserv.proto.Graph.TEdge;
import crypta.idserv.proto.Graph.TGraph;
import crypta.idserv.proto.Graph.TNode;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.VultureGraphSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.misc.lang.number.UnsignedLong;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.crypta.graph.api.model.ids.GraphId.CRYPTA_ID_TYPE;

public class VultureGraphService implements GraphService {
    private YtClient yt;
    private VultureGraphSettings settings;

    @Inject
    public VultureGraphService(YtService yt, VultureGraphSettings settings) {
        this.settings = settings;
        this.yt = yt.getSenecaRpc();
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams searchParams, InfoParams infoParams) {
        String actualCryptaId;

        if (CRYPTA_ID_TYPE.equals(id.getIdType())) {
            actualCryptaId = id.getIdValue();
        } else {
            actualCryptaId = mapIdToCryptaId(id, searchParams.getMatchingType());
        }

        if (actualCryptaId == null) {
            return Optional.empty();
        }

        return getByCryptaId(actualCryptaId, searchParams.getMatchingType());
    }

    private String mapIdToCryptaId(GraphId id, String matchType) {
        TableSchema schema = new TableSchema.Builder()
                .addKey("IdType", ColumnValueType.STRING)
                .addKey("IdValue", ColumnValueType.STRING)
                .addValue("CryptaId", ColumnValueType.UINT64)
                .build();

        var results = YtReadingUtils.lookupRows(
                yt,
                settings.getIdToCryptaIdTable(matchType),
                schema,
                List.of(id.getIdType(), id.getIdValue()),
                settings.getLookupTimeoutMilliseconds()
        );

        if (results.size() > 0) {
            return Long.toUnsignedString(results.get(0).getLong("CryptaId"));
        } else {
            return null;
        }
    }

    @Override
    public GraphSettings getGraphSettings() {
        return settings;
    }

    private Optional<Graph> getByCryptaId(String cryptaId, String matchType) {
        List<TGraph> graphs = new ArrayList<>();

        TableSchema schema = new TableSchema.Builder()
                .addKey("CryptaId", ColumnValueType.UINT64)
                .addValue("Graph", ColumnValueType.STRING)
                .build();

        var results = YtReadingUtils.lookupRows(
                yt,
                settings.getCryptaIdToGraph(matchType),
                schema,
                List.of(Long.parseUnsignedLong(cryptaId)),
                settings.getLookupTimeoutMilliseconds()
        );

        for (YTreeMapNode record : results) {
            try {
                graphs.add(TGraph.parseFrom(record.getBytes("Graph")));
            } catch (InvalidProtocolBufferException e) {
                throw Exceptions.internal("Invalid protocol buffer");
            }
        }

        if (graphs.isEmpty()) {
            return Optional.empty();
        }

        TGraph graphFromProto = graphs.get(0);

        String cryptaId2 = UnsignedLong.valueOf(graphFromProto.getCryptaId()).toString();

        List<Vertex> vertices = graphFromProto.getNodesList().stream()
                .map(this::mapVertexNode).collect(Collectors.toList());

        List<Edge> edges = graphFromProto.getEdgesList().stream()
                .map(edgeNode -> mapEdgeNode(vertices, edgeNode))
                .collect(Collectors.toList());

        GraphComponent graphComponent = new GraphComponent(cryptaId2, vertices, edges);

        return Optional.of(new Graph(List.of(graphComponent)));
    }

    private Vertex mapVertexNode(TNode node) {
        return new Vertex(node.getId(), node.getType());
    }

    private Edge mapEdgeNode(List<Vertex> vertices, TEdge edgeNode) {
        int v1Id = edgeNode.getNode1();
        Vertex vertex1 = vertices.get(v1Id);

        int v2Id = edgeNode.getNode2();
        Vertex vertex2 = vertices.get(v2Id);

        Option<List<TAttribute>> edgeAttrs = Option.of(edgeNode.getAttributesList());

        String sourceType = "unknown";
        String logSource = "unknown";
        List<String> attrs = new ArrayList<>();

        if (edgeAttrs.isPresent()) {
            Map<String, String> attrsMap = edgeAttrs.get()
                    .stream()
                    .collect(Collectors.toMap(
                            TAttribute::getName,
                            TAttribute::getValue
                    ));
            sourceType = attrsMap.getOrDefault("source_type", sourceType);
            logSource = attrsMap.getOrDefault("log_source", logSource);
            if (attrsMap.get("indevice") != null) {
                attrs.add("indevice");
            }
            if (attrsMap.get("realtime") != null) {
                attrs.add("realtime");
            }
        }

        Edge edge = new Edge(vertex1.getIdValue(), vertex1.getIdType(),
                vertex2.getIdValue(), vertex2.getIdType(),
                sourceType,
                logSource,
                0.0,
                Cf.list()
        );
        edge.setAttributes(attrs);
        return edge;
    }
}
