package ru.yandex.crypta.graph.api.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.inject.Inject;

import com.google.protobuf.InvalidProtocolBufferException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponentWithInfo;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.engine.GraphEngineConverter;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.YtGraphEngineSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.graph.engine.proto.TGraph;
import ru.yandex.crypta.lib.yt.YsonMapper;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static java.util.stream.Collectors.toList;
import static ru.yandex.crypta.graph.api.model.ids.GraphId.CRYPTA_ID_TYPE;
import static ru.yandex.crypta.graph.api.model.ids.GraphId.MERGE_KEY;

public class YtGraphEngineService implements GraphService {

    private static final Logger LOG = LoggerFactory.getLogger(YtGraphEngineService.class);

    private YtService yt;
    private YtGraphEngineSettings graphSettings;
    GraphEngineConverter graphEngine;

    @Inject
    public YtGraphEngineService(YtService yt, YtGraphEngineSettings graphSettings) {
        this.yt = yt;
        this.graphSettings = graphSettings;
        this.graphEngine = new GraphEngineConverter();
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams params, InfoParams infoParams) {

        if (CRYPTA_ID_TYPE.equals(id.getIdType())) {
            return getByCryptaId(id.getIdValue());
        } else if (MERGE_KEY.equals(id.getIdType())) {
            return getByMergeKey(id.getIdValue());
        } else {
            return getByUsualId(id);
        }
    }

    private Optional<Graph> getByCryptaId(String cryptaId) {
        YPath byIdPath = graphSettings.getPaths().byCryptaId(cryptaId);

        Optional<TGraph> maybeTGraph = yt.readTableYson(byIdPath, (recs) -> {
            byte[] graphBytes = recs.getBytes("graph");
            try {
                return TGraph.parseFrom(graphBytes);
            } catch (InvalidProtocolBufferException e) {
                throw new IllegalStateException(e);
            }
        }).stream().findFirst();

        return maybeTGraph.map(tGraph -> new Graph(graphEngine.convert(tGraph)));
    }

    private YsonMapper<String> cryptaIdMapper = (rec) -> rec.getString("cryptaId");

    private Optional<Graph> getByUsualId(GraphId id) {
        YPath byIdPath = graphSettings.getPaths().byId(id);

        LOG.info("Fetching id->cryptaId from " + byIdPath.toString());
        Optional<String> maybeCryptaId = yt.readTableYson(byIdPath, cryptaIdMapper).stream().findFirst();

        return maybeCryptaId.flatMap(this::getByCryptaId);
    }

    private Optional<Graph> getByMergeKey(String mergeKey) {
        YPath byIdPath = graphSettings.getPaths().byMergeKey(mergeKey);

        List<TGraph> components = new ArrayList<>(
                yt.readTableYson(byIdPath, (rec) -> {
                    byte[] graphBytes = rec.getBytes("graphEngine");
                    try {
                        TGraph tGraph = TGraph.parseFrom(graphBytes);
                        long cryptaId = Long.parseUnsignedLong(rec.getString("cryptaId"));
                        return tGraph.toBuilder()
                                .setId(cryptaId)
                                .build();
                    } catch (InvalidProtocolBufferException e) {
                        throw new IllegalStateException(e);
                    }
                })
        );

        YPath edgesBetweenPath = graphSettings.getPaths().edgesBetweenByMergeKey(mergeKey);
        List<Edge> edgesBetween = new ArrayList<>(
                yt.readTableYson(edgesBetweenPath, YtHumanMatchingGraphService.EDGE_MAPPER)
        );

        if (components.isEmpty()) {
            return Optional.empty();
        } else {
            List<GraphComponentWithInfo> subGraphs = components
                    .stream()
                    .map(graphEngine::convert)
                    .collect(toList());

            Graph result = new Graph();
            for (GraphComponentWithInfo subGraph : subGraphs) {
                result.addComponentWithInfo(subGraph);
            }
            result.setEdgesBetweenComponents(edgesBetween);
            return Optional.of(result);
        }
    }


    @Override
    public GraphSettings getGraphSettings() {
        return new YtGraphEngineSettings();
    }
}
