package ru.yandex.crypta.graph.api.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.inject.Inject;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.graph.GraphComponentWithInfo;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.YtHouseholdsGraphSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.lib.yt.YsonMapper;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static ru.yandex.crypta.graph.api.model.ids.GraphId.YANDEXUID_TYPE;

public class YtHouseholdsGraphService implements GraphService {
    private YtService yt;

    private YtHouseholdsGraphSettings settings;

    private YsonMapper<String> hhIdMapper = (rec) -> rec.getString("hhid");

    private YsonMapper<Edge> edgeMapper = (rec) -> {
        final String id1Type = GraphId.CRYPTA_ID_TYPE;
        String id2Type;
        if (rec.getBool("is_tv")) {
            id2Type = YANDEXUID_TYPE + "_tv";
        } else {
            id2Type = YANDEXUID_TYPE;
        }

        return new Edge(
                rec.getStringO("crypta_id").orElse("None"),
                id1Type,
                rec.getString("yuid"),
                id2Type,
                "households",
                "",
                0.0,
                Cf.list()
        );
    };

    @Inject
    public YtHouseholdsGraphService(YtService yt, YtHouseholdsGraphSettings settings) {
        this.yt = yt;
        this.settings = settings;
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams params, InfoParams infoParams) {
        if (id.getIdType().equals(GraphId.HOUSEHOLD_ID_TYPE)) {
            return getGraphByHouseholdId(id.getIdValue());
        } else {
            YPath reversedPath = settings.getPaths().getReversedPath(id);
            ListF<String> hhIds = yt.readTableYson(reversedPath, hhIdMapper);
            if (hhIds.isEmpty()) {
                return Optional.empty();
            } else {
                String cryptaId = hhIds.first();
                return getGraphByHouseholdId(cryptaId);
            }
        }
    }

    private Optional<Graph> getGraphByHouseholdId(String hhId) {
        YPath enrichPath = settings.getPaths().getEnrichPath(hhId);
        ListF<Edge> edges = yt.readTableYson(enrichPath, edgeMapper);

        if (edges.isEmpty()) {
            return Optional.empty();
        } else {
            List<Vertex> vertices = new ArrayList<>(GraphComponent.verticesOfEdges(edges));
            GraphComponent mainComponent = new GraphComponent(hhId, vertices, edges);
            return Optional.of(new Graph(new GraphComponentWithInfo(mainComponent)));
        }
    }

    @Override
    public GraphSettings getGraphSettings() {
        return settings;
    }
}
