package ru.yandex.crypta.graph.api.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.protobuf.InvalidProtocolBufferException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.clients.utils.JsonUtils;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.graph.GraphComponentInfo;
import ru.yandex.crypta.graph.api.model.graph.GraphComponentWithInfo;
import ru.yandex.crypta.graph.api.model.graph.MergeInfo;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.model.ids.GraphIdInfo;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.YtHumanMatchingGraphSettings;
import ru.yandex.crypta.graph.api.service.settings.YtHumanMatchingGraphSettings.GraphPaths;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.graph2.dao.yt.utils.YTreeUtils;
import ru.yandex.crypta.graph2.model.matching.component.similarity.SimilarityResult;
import ru.yandex.crypta.graph2.model.matching.merge.MergeKey;
import ru.yandex.crypta.graph2.model.matching.proto.SplitInfo;
import ru.yandex.crypta.graph2.model.matching.score.MetricsTree;
import ru.yandex.crypta.lib.yt.JsonMapper;
import ru.yandex.crypta.lib.yt.JsonMultiMapper;
import ru.yandex.crypta.lib.yt.YsonMapper;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeDoubleNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.crypta.graph.api.model.ids.GraphId.CRYPTA_ID_TYPE;
import static ru.yandex.crypta.graph.api.service.settings.YtHumanMatchingGraphSettings.MATCH_SCOPE_NEIGHBOURS;

public class YtHumanMatchingGraphService implements GraphService {

    private static final Logger LOG = LoggerFactory.getLogger(YtHumanMatchingGraphService.class);

    private YtService yt;
    private YtHumanMatchingGraphSettings graphSettings;

    public static final YsonMapper<Edge> EDGE_MAPPER = (rec) -> {
        Double survivalWeight = YTreeUtils.<YTreeDoubleNode>getNullableNode(rec, "survivalWeight")
                .map(YTreeDoubleNode::getValue)
                .orElse(null);

        Optional<YTreeNode> maybeDates = YTreeUtils.getNullableNode(rec, "dates");
        List<String> dates = maybeDates.stream()
                .flatMap(y -> y.asList().stream().map(YTreeNode::stringValue))
                .collect(toList());
        Edge edge = new Edge(
                rec.getString("id1"),
                rec.getString("id1Type"),
                rec.getString("id2"),
                rec.getString("id2Type"),
                rec.getString("sourceType"),
                rec.getString("logSource"),
                survivalWeight,
                dates,
                rec.getStringO("merge_key").orElse("")
        );

        Optional<YTreeNode> indeviceNode = YTreeUtils.getNullableNode(rec, "indevice");
        if (indeviceNode.isPresent() && indeviceNode.get().boolValue()) {
            edge.addAttribute("indevice");
        }

        return edge;
    };

    private YsonMapper<String> cryptaIdMapper = (rec) -> rec.getString("cryptaId");

    private YsonMapper<Vertex> vertexMapper = (rec) ->
            new Vertex(
                    rec.getString("id"),
                    rec.getString("id_type")
            );

    private YsonMapper<List<String>> neighboursMapper = (rec) -> {
        YTreeNode neighbours = rec.getOrThrow("neighbours");

        return neighbours.asList().stream().map(YTreeNode::stringValue).collect(Collectors.toList());
    };

    private YsonMapper<GraphComponentInfo> componentInfoMapper = (rec) ->
            new GraphComponentInfo(
                    rec.getString("cryptaId"),
                    new GraphId(rec.getString("ccId"), rec.getString("ccIdType")),
                    rec.getInt("verticesCount"),
                    rec.getInt("ccNeighboursCount"),
                    Cf.wrap(rec.get("score").get().asMap())
            );

    private JsonMapper<GraphId> graphIdMapper = (rec) ->
            new GraphId(rec.get("id").textValue(), rec.get("id_type").textValue());

    private JsonMultiMapper<GraphIdInfo> idsInfoMapper = (recs) ->
            Cf.wrap(recs).groupBy(graphIdMapper)
                    .mapEntries((id, infoRecs) -> {
                        GraphIdInfo graphIdInfo = new GraphIdInfo(id);
                        for (JsonNode infoRec : infoRecs) {
                            Map<String, JsonNode> idProps = JsonUtils.jsonToMap(infoRec, r -> r);
                            graphIdInfo.addInfoRec(idProps);
                        }

                        return graphIdInfo;
                    });

    private YsonMapper<MergeInfo> mergeInfoMapper = (rec) -> {
        Map<String, YTreeNode> scoreGain = new HashMap<>();
        if (rec.get("scoreGain").get().isMapNode()) {
            scoreGain = rec.get("scoreGain").get().asMap();
        }
        return new MergeInfo(
                rec.getString("merge_key"),
                rec.getString("status"),
                rec.getString("strength"),
                rec.getDouble("weight"),
                Cf.wrap(scoreGain),
                Cf.wrap(rec.get("similarityResults").get().asList())
        );
    };

    private YsonMapper<List<MergeInfo>> splitInfoMapper = (rec) -> {
        byte[] bytes = rec.getBytes("offers");
        List<SplitInfo.InnerSplitOffer> splitOffers;
        try {
            splitOffers = SplitInfo.InnerSplitOffers.parseFrom(bytes).getValuesList();
        } catch (InvalidProtocolBufferException e) {
            throw Exceptions.unchecked(e);
        }

        return splitOffers.stream().map(splitOffer -> {
            MergeKey mergeKey = MergeKey.betweenComponents(
                    splitOffer.getLeftCryptaId(),
                    splitOffer.getRightCryptaId()
            );

            Map<String, Double> innerScores = splitOffer.getScores()
                    .getValuesList()
                    .stream()
                    .collect(toMap(
                            SplitInfo.InnerSplitOffer.Score::getName,
                            SplitInfo.InnerSplitOffer.Score::getValue
                    ));
            MetricsTree metricsTree = new MetricsTree(
                    splitOffer.getScore(),
                    Cf.wrap(innerScores)
            );

            List<SimilarityResult> similarityResults = splitOffer
                    .getSimilarity()
                    .getFailedSimilarityTypesList()
                    .stream().map(sim -> new SimilarityResult(false, sim))
                    .collect(toList());

            return new MergeInfo(
                    mergeKey.getMergeKey(),
                    splitOffer.getStatus(),
                    "",
                    splitOffer.getEdgesScore(),
                    metricsTree,
                    similarityResults
            );
        }).collect(toList());

    };

    @Inject
    public YtHumanMatchingGraphService(YtService yt, YtHumanMatchingGraphSettings graphSettings) {
        this.yt = yt;
        this.graphSettings = graphSettings;
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams searchParams, InfoParams infoParams) {
        if (CRYPTA_ID_TYPE.equals(id.getIdType())) {
            return getByCryptaId(id.getIdValue(), searchParams, infoParams);
        } else {
            return getByUsualId(id, searchParams, infoParams);
        }
    }

    @Override
    public GraphSettings getGraphSettings() {
        return graphSettings;
    }

    private Optional<Graph> getByUsualId(GraphId id, SearchParams params, InfoParams infoParams) {
        GraphPaths paths = graphSettings.getPaths(params.getMatchingType());

        YPath pathWithKey = paths.getVerticesPath(id);
        Option<String> maybeCryptaId = yt.readTableYson(pathWithKey, cryptaIdMapper).firstO();

        LOG.info("Fetching id->cryptaId from " + pathWithKey.toString());

        return maybeCryptaId.flatMapO(cryptaId -> Option.wrap(
                getByCryptaId(cryptaId, params, infoParams)
        )).toOptional();
    }


    private Optional<Graph> getByCryptaId(String cryptaId, SearchParams params, InfoParams infoParams) {
        GraphPaths paths = graphSettings.getPaths(params.getMatchingType());
        if (MATCH_SCOPE_NEIGHBOURS.equals(params.getMatchingScope())) {
            return getByCryptaIdWithNeighbours(cryptaId, infoParams, paths);

        } else {

            return getByCryptaIdSingle(cryptaId, infoParams, paths);
        }

    }

    private Optional<Graph> getByCryptaIdSingle(String cryptaId, InfoParams infoParams, GraphPaths paths) {
        GraphComponentWithInfo singleComponent = getSingleComponent(cryptaId, paths, infoParams);
        if (singleComponent.isEmpty()) {
            return Optional.empty();
        } else {
            Graph graph = new Graph(singleComponent);
            return Optional.of(graph);
        }
    }

    private Optional<Graph> getByCryptaIdWithNeighbours(String cryptaId, InfoParams infoParams, GraphPaths paths) {
        YPath neighboursTablePath = paths.getGraphNeighboursPath(cryptaId);

        GraphComponentWithInfo mainComponent = getSingleComponent(cryptaId, paths, infoParams.withNeighbours());

        List<String> neighbourCryptaIds =
                yt.readTableYson(neighboursTablePath, neighboursMapper)
                        .firstO()
                        .<String>flatten()
                        .unique().toList();

        List<CompletableFuture<GraphComponentWithInfo>> neighboursResults = neighbourCryptaIds
                .stream()
                .map(neighbour -> CompletableFuture.supplyAsync(
                        () -> getSingleComponent(neighbour, paths, infoParams.forNeighbourComponent())
                )).collect(toList());

        CompletableFuture.allOf(neighboursResults.toArray(new CompletableFuture[0])).join();
        List<GraphComponentWithInfo> neighbourComponents = neighboursResults
                .stream()
                .map(CompletableFuture::join)
                .filter(c -> !c.isEmpty())
                .collect(toList());

        if (mainComponent.isEmpty()) {
            return Optional.empty();
        } else {
            Graph graph = new Graph(mainComponent);
            for (GraphComponentWithInfo component : neighbourComponents) {
                graph.addComponentWithInfo(component);
            }
            return Optional.of(graph);
        }
    }


    private GraphComponentWithInfo getSingleComponent(String cryptaId, GraphPaths paths, InfoParams infoParams) {

        CompletableFuture<List<Edge>> edgesResult = getEdges(cryptaId, paths);
        CompletableFuture<Optional<GraphComponentInfo>> componentInfoResult = getComponentInfo(cryptaId, paths,
                infoParams);
        CompletableFuture<List<GraphIdInfo>> idsInfoResult = getIdsInfo(cryptaId, paths, infoParams);
        CompletableFuture<List<MergeInfo>> mergeInfoResult = getMergeInfo(cryptaId, paths, infoParams);
        CompletableFuture<List<MergeInfo>> splitInfoResult = getSplitInfo(cryptaId, paths, infoParams);
        CompletableFuture<List<Edge>> edgesBetweenResult = getEdgesBetween(cryptaId, paths, infoParams);

        CompletableFuture.allOf(
                edgesResult,
                componentInfoResult,
                idsInfoResult,
                mergeInfoResult,
                splitInfoResult,
                edgesBetweenResult
        ).join();


        List<Edge> edges = edgesResult.join();
        List<Vertex> vertices;
        if (edges.isEmpty()) {
            // case when component contains from single vertex
            vertices = yt.readTableYson(paths.getVerticesByCryptaIdPath(cryptaId), vertexMapper);
        } else {
            vertices = new ArrayList<>(GraphComponent.verticesOfEdges(edges));
        }

        GraphComponent graphComponent = new GraphComponent(cryptaId, vertices, edges);
        GraphComponentWithInfo withInfo = new GraphComponentWithInfo(graphComponent);

        Optional<GraphComponentInfo> componentInfo = componentInfoResult.join();
        componentInfo.ifPresent(withInfo::setComponentsInfo);

        List<MergeInfo> mergeInfo = new ArrayList<>();
        mergeInfo.addAll(mergeInfoResult.join());
        mergeInfo.addAll(splitInfoResult.join());
        withInfo.setMergeInfo(mergeInfo);

        withInfo.setIdsInfo(idsInfoResult.join());
        withInfo.setEdgesBetweenComponents(edgesBetweenResult.join());

        return withInfo;

    }

    private CompletableFuture<List<Edge>> getEdges(String cryptaId, GraphPaths paths) {
        return CompletableFuture.supplyAsync(() -> yt.readTableYson(paths.getEdgesPath(cryptaId), EDGE_MAPPER));
    }

    private CompletableFuture<Optional<GraphComponentInfo>> getComponentInfo(String cryptaId, GraphPaths paths,
                                                                             InfoParams infoParams) {
        if (infoParams.includeComponentInfo) {
            YPath graphStatsPath = paths.getGraphStatsPath(cryptaId);
            return CompletableFuture.supplyAsync(() -> yt.readTableYson(graphStatsPath, componentInfoMapper).firstO().toOptional());
        } else {
            return CompletableFuture.completedFuture(Optional.empty());
        }
    }

    private CompletableFuture<List<GraphIdInfo>> getIdsInfo(String cryptaId, GraphPaths paths, InfoParams infoParams) {
        if (infoParams.includeIdsInfo) {
            YPath verticesPropertiesPath = paths.getVerticesPropertiesPath(cryptaId);
            List<JsonNode> idInfoRecs = yt.readTableJson(verticesPropertiesPath, (r) -> r);
            return CompletableFuture.supplyAsync(() -> idsInfoMapper.apply(idInfoRecs));
        } else {
            return CompletableFuture.completedFuture(List.of());
        }
    }

    private CompletableFuture<List<MergeInfo>> getMergeInfo(String cryptaId, GraphPaths paths, InfoParams infoParams) {
        if (infoParams.includeMergeInfo) {
            YPath mergeOffersPath = paths.getMergeOffersPath(cryptaId);
            return CompletableFuture.supplyAsync(() -> yt.readTableYson(mergeOffersPath, mergeInfoMapper));
        } else {
            return CompletableFuture.completedFuture(List.of());
        }
    }

    private CompletableFuture<List<MergeInfo>> getSplitInfo(String cryptaId, GraphPaths paths, InfoParams infoParams) {
        if (infoParams.includeMergeInfo) {
            YPath splitInfoPath = paths.getSplitInfoPath(cryptaId);
            return CompletableFuture.supplyAsync(() -> yt.readTableYson(splitInfoPath, splitInfoMapper)
                    .stream()
                    .flatMap(Collection::stream)
                    .collect(toList())
            );
        } else {
            return CompletableFuture.completedFuture(List.of());
        }
    }

    private CompletableFuture<List<Edge>> getEdgesBetween(String cryptaId, GraphPaths paths, InfoParams infoParams) {
        if (infoParams.includeEdgesBetween) {
            YPath edgesBetweenPath = paths.getEdgesBetweenPath(cryptaId);
            return CompletableFuture.supplyAsync(() -> yt.readTableYson(edgesBetweenPath, EDGE_MAPPER));
        } else {
            return CompletableFuture.completedFuture(List.of());
        }

    }

}
