package ru.yandex.crypta.graph.api.service;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import com.google.protobuf.InvalidProtocolBufferException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.YtProtoExportGraphSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.lang.number.UnsignedLong;

import static ru.yandex.crypta.graph.api.service.settings.YtProtoExportGraphSettings.ERRORS_MATCH_SCOPE;
import static ru.yandex.crypta.graph.api.service.settings.YtProtoExportGraphSettings.ERRORS_PATH;
import static ru.yandex.crypta.graph.api.service.settings.YtProtoExportGraphSettings.OK_PATH;
import static ru.yandex.crypta.graph.api.service.settings.YtProtoExportGraphSettings.SPANNING_MATCH_SCOPE;

public class YtProtoExportGraphService implements GraphService {

    private GraphSettings graphSettings;
    private YtService yt;

    @Inject
    public YtProtoExportGraphService(YtProtoExportGraphSettings graphSettings, YtService yt) {
        this.graphSettings = graphSettings;
        this.yt = yt;
    }

    /**
     * @deprecated replace with Optional.stream in Java 9+
     */
    @Deprecated
    private static <T> Stream<T> optionalToStream(Optional<T> opt) {
        return opt.map(Stream::of).orElseGet(Stream::empty);
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams params, InfoParams infoParams) {

        if (!GraphId.CRYPTA_ID_TYPE.equals(id.getIdType())) {
            throw new IllegalStateException("Can only fetch graph by crypta_id");
        }

        boolean onlySpanning = SPANNING_MATCH_SCOPE.equals(params.getMatchingScope());

        UnsignedLong cryptaIdKey = UnsignedLong.valueOf(Long.parseUnsignedLong(id.getIdValue()));
        YPath path = ERRORS_MATCH_SCOPE.equals(params.getMatchingScope()) ? ERRORS_PATH : OK_PATH;
        YPath pathByKey = path.withExact(YtReadingUtils.exact(cryptaIdKey));

        ListF<GraphComponent> graphComponents = yt.readTableYson(pathByKey, (rec) -> {
            UnsignedLong cryptaId = UnsignedLong.valueOf(rec.getLong("cryptaId"));
            byte[] graphBinary = rec.getBytes("graph");
            List<Edge> edges;
            try {
                crypta.idserv.proto.Graph.TGraph protoGraph = crypta.idserv.proto.Graph.TGraph.parseFrom(graphBinary);
                List<crypta.idserv.proto.Graph.TNode> vertices = protoGraph.getNodesList();

                edges = protoGraph.getEdgesList()
                        .stream()
                        .flatMap(protoEdge ->
                                optionalToStream(mapProtoEdge(protoEdge, vertices, onlySpanning)))
                        .collect(Collectors.toList());

            } catch (InvalidProtocolBufferException e) {
                throw Exceptions.internal(e.getMessage());
            }
            return new GraphComponent(cryptaId.toString(), Cf.toList(GraphComponent.verticesOfEdges(edges)), edges);
        });

        return Optional.of(new Graph(graphComponents));
    }

    private Optional<Edge> mapProtoEdge(crypta.idserv.proto.Graph.TEdge protoEdge,
                                        List<crypta.idserv.proto.Graph.TNode> vertices, boolean onlySpanning) {
        crypta.idserv.proto.Graph.TNode node1 = vertices.get(protoEdge.getNode1());
        crypta.idserv.proto.Graph.TNode node2 = vertices.get(protoEdge.getNode2());
        List<crypta.idserv.proto.Graph.TAttribute> attrs = protoEdge.getAttributesList();

        String spanning = extractAttributeValue(attrs, "spanning");
        boolean isSpanning = spanning.equals("y");
        if (onlySpanning && !isSpanning) {
            return Optional.empty();

        } else {
            Edge edge = new Edge(
                    node1.getId(),
                    node1.getType(),
                    node2.getId(),
                    node2.getType(),
                    extractAttributeValue(attrs, "source_type"),
                    extractAttributeValue(attrs, "log_source"),
                    0.0,
                    Cf.list()
            );
            if (isSpanning) {
                edge.addAttribute("spanning");
            }
            return Optional.of(edge);
        }

    }

    private String extractAttributeValue(List<crypta.idserv.proto.Graph.TAttribute> attributesList, String attr) {
        Stream<crypta.idserv.proto.Graph.TAttribute> attrs = attributesList.stream();
        return attrs.filter(a -> a.getName().equals(attr)).findFirst().map(
                crypta.idserv.proto.Graph.TAttribute::getValue
        ).orElse("");
    }

    @Override
    public GraphSettings getGraphSettings() {
        return graphSettings;
    }
}
