package ru.yandex.crypta.graph.api.service;

import java.util.Optional;

import javax.inject.Inject;

import com.google.protobuf.InvalidProtocolBufferException;
import org.jetbrains.annotations.NotNull;

import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.engine.GraphEngineConverter;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.YtRTGraphEngineSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.graph.engine.proto.TGraph;
import ru.yandex.crypta.graph.rt.sklejka.michurin.proto.TMichurinState;
import ru.yandex.crypta.lib.proto.identifiers.TGenericID;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.service.bigrt.BigRTHelper;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeKeyField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeSerializerClass;
import ru.yandex.inside.yt.kosher.impl.ytree.object.serializers.YTreeBytesToStringSerializer;
import ru.yandex.inside.yt.kosher.impl.ytree.object.serializers.YTreeObjectSerializer;
import ru.yandex.yt.ytclient.proxy.MappedLookupRowsRequest;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

public class YtRTGraphEngineService implements GraphService {

    private final YtService yt;

    private final YtRTGraphEngineSettings settings;
    private final GraphEngineConverter graphEngine;
    private final BigRTHelper bigRTHelper;

    @Inject
    public YtRTGraphEngineService(YtService yt, YtRTGraphEngineSettings settings) {
        this.yt = yt;
        this.settings = settings;
        this.graphEngine = new GraphEngineConverter();
        this.bigRTHelper = new BigRTHelper();
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams params, InfoParams infoParams) {

        if (id.isCryptaId()) {
            return getByCryptaId(id.getIdValue(), params.getMatchingType());
        } else {
            return getById(id, params.getMatchingType());
        }
    }

    @SuppressWarnings("UnusedVariable")
    @YTreeObject
    public static class IdToCryptaIdRow {
        @YTreeField(key = "Id")
        @YTreeKeyField
        @YTreeSerializerClass(YTreeBytesToStringSerializer.class)
        private byte[] id;

        @YTreeField(key = "CryptaId")
        @YTreeSerializerClass(YTreeBytesToStringSerializer.class)
        private byte[] cryptaId;

        public IdToCryptaIdRow(byte[] id) {
            this.id = id;
        }
    }

    private Optional<Graph> getById(GraphId id, String matchType) {
        // use lookup here as it's hard to query byte[] key using selectRows
        MappedLookupRowsRequest<IdToCryptaIdRow> request = getIdToCryptaIdRequest(matchType, id);

        Optional<TGenericID> maybeCryptaId = yt.getHahnRpc().lookupRows(request)
                .thenApply(rows ->
                        rows.getYTreeRows()
                                .stream()
                                .findFirst()
                                .map(row -> row.getBytes("CryptaId"))
                                .flatMap(data -> {
                                    try {
                                        return Optional.of(TGenericID.parseFrom(data));
                                    } catch (InvalidProtocolBufferException e) {
                                        return Optional.empty();
                                    }
                                })

                ).join();

        return maybeCryptaId.map(TGenericID::getCryptaId)
                .map(typedCryptaId -> Long.toUnsignedString(typedCryptaId.getValue()))
                .flatMap(cryptaId -> getByCryptaId(cryptaId, matchType));
    }

    @NotNull
    private MappedLookupRowsRequest<IdToCryptaIdRow> getIdToCryptaIdRequest(String matchType, GraphId id) {
        Identifier typedId = new Identifier(id.getIdType(), id.getIdValue());
        MappedLookupRowsRequest<IdToCryptaIdRow> request = new MappedLookupRowsRequest<>(
                settings.getCryptaIdIndexPath(matchType).toString(),
                new YTreeObjectSerializer<>(IdToCryptaIdRow.class)
        );
        request.addAllLookupColumns();
        request.addFilter(new IdToCryptaIdRow(typedId.toProto()));
        return request;
    }

    private Optional<Graph> getByCryptaId(String cryptaId, String matchType) {
        String query = String.format(
                "* FROM [%s] WHERE Id = %s LIMIT 1",
                settings.getMichurinStatePath(matchType),
                cryptaId
        );

        return yt.getHahnRpc().selectRows(query).thenApply(this::fetchGraph).join();
    }

    private Optional<Graph> fetchGraph(UnversionedRowset rowSet) {
        return rowSet
                .getYTreeRows()
                .stream()
                .findFirst()
                .map(row -> bigRTHelper.parseState(
                        row.getBytes("State"),
                        TMichurinState.parser()
                ))
                .map(this::toGraph);
    }

    private Graph toGraph(TMichurinState state) {
        TGraph graph = state.getGraph();
        return new Graph(graphEngine.convert(graph));
    }

    @Override
    public GraphSettings getGraphSettings() {
        return settings;
    }
}
