package ru.yandex.crypta.graph.api.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.inject.Inject;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.graph.GraphComponentWithInfo;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.YtSoupyIndeviceGraphSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.lib.yt.YsonMapper;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class YtSoupyIndeviceGraphService implements GraphService {
    private YtService yt;

    private YtSoupyIndeviceGraphSettings settings;

    private YsonMapper<String> indeviceIdMapper = (rec) -> rec.getString("indevice_id");

    private YsonMapper<Edge> recMapper = (rec) -> {
        return new Edge(
                rec.getString("indevice_id"),
                "indevice_id",
                rec.getString("id"),
                rec.getString("id_type"),
                "soupy",
                "indevice",
                0.0,
                Cf.list(
                        rec.getStringO("date_begin").orElse("unknown_begin_date"),
                        rec.getStringO("date_end").orElse("unknown_end_date")
                )
        );
    };

    @Inject
    public YtSoupyIndeviceGraphService(YtService yt, YtSoupyIndeviceGraphSettings settings) {
        this.yt = yt;
        this.settings = settings;
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams params, InfoParams infoParams) {
        if (id.getIdType().equals(GraphId.HOUSEHOLD_ID_TYPE)) {
            return getGraphByIndeviceId(id.getIdValue());
        } else {
            return getByUsualId(id);
        }
    }

    private Optional<Graph> getByUsualId(GraphId id) {
        YPath indexPath = settings.getPaths().byId(id);
        ListF<String> indeviceId = yt.readTableYson(indexPath, indeviceIdMapper);
        if (indeviceId.isEmpty()) {
            return Optional.empty();
        } else {
            String cryptaId = indeviceId.first();
            return getGraphByIndeviceId(cryptaId);
        }
    }

    private Optional<Graph> getGraphByIndeviceId(String indeviceId) {
        YPath byIndeviceIdPath = settings.getPaths().byIndeviceId(indeviceId);
        ListF<Edge> recs = yt.readTableYson(byIndeviceIdPath, recMapper);

        if (recs.isEmpty()) {
            return Optional.empty();
        } else {
            List<Vertex> vertices = new ArrayList<>(GraphComponent.verticesOfEdges(recs));
            GraphComponent mainComponent = new GraphComponent(indeviceId, vertices, recs);
            return Optional.of(new Graph(new GraphComponentWithInfo(mainComponent)));
        }
    }

    @Override
    public GraphSettings getGraphSettings() {
        return settings;
    }
}
