package ru.yandex.crypta.graph.api.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.inject.Inject;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.graph.GraphComponentWithInfo;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.settings.GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.YtV1GraphSettings;
import ru.yandex.crypta.graph.api.service.settings.model.InfoParams;
import ru.yandex.crypta.graph.api.service.settings.model.SearchParams;
import ru.yandex.crypta.lib.yt.YsonMapper;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class YtV1GraphService implements GraphService {
    private YtService yt;

    private YtV1GraphSettings settings;

    private YsonMapper<String> cryptaIdMapper = (rec) -> rec.getString("crypta_id");

    private YsonMapper<Edge> edgeMapper = (rec) -> {
        String id1Type;
        String id2Type;
        if ("d_y".equals(rec.getString("pair_type"))) {
            id1Type = GraphId.OLD_DEVICE_ID;
            id2Type = GraphId.YANDEXUID_TYPE;
        } else {
            id1Type = GraphId.YANDEXUID_TYPE;
            id2Type = GraphId.YANDEXUID_TYPE;
        }

        return new Edge(
                rec.getString("id1"),
                id1Type,
                rec.getString("id2"),
                id2Type,
                rec.getString("pair_source"),
                "",
                0.0,
                Cf.list());
    };

    @Inject
    public YtV1GraphService(YtService yt, YtV1GraphSettings settings) {
        this.yt = yt;
        this.settings = settings;
    }

    @Override
    public Optional<Graph> getById(GraphId id, SearchParams params, InfoParams infoParams) {
        if (id.getIdType().equals(GraphId.OLD_CRYPTA_ID_TYPE)) {
            return getGraphByCryptaId(id.getIdValue());
        } else {
            YPath verticesPath = settings.getPaths().getVerticesPath(id);
            ListF<String> cryptaIds = yt.readTableYson(verticesPath, cryptaIdMapper);
            if (cryptaIds.isEmpty()) {
                return Optional.empty();
            } else {
                String cryptaId = cryptaIds.first();
                return getGraphByCryptaId(cryptaId);
            }
        }
    }

    private Optional<Graph> getGraphByCryptaId(String cryptaId) {
        YPath edgesPath = settings.getPaths().getEdgesPath(cryptaId);
        ListF<Edge> edges = yt.readTableYson(edgesPath, edgeMapper);

        if (edges.isEmpty()) {
            return Optional.empty();
        } else {
            List<Vertex> vertices = new ArrayList<>(GraphComponent.verticesOfEdges(edges));
            GraphComponent mainComponent = new GraphComponent(cryptaId, vertices, edges);

            return Optional.of(new Graph(new GraphComponentWithInfo(mainComponent)));
        }
    }

    @Override
    public GraphSettings getGraphSettings() {
        return settings;
    }
}
