package ru.yandex.crypta.graph.api.service.engine;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.util.JsonFormat;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.crypta.clients.utils.JsonUtils;
import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.graph.GraphComponentInfo;
import ru.yandex.crypta.graph.api.model.graph.GraphComponentWithInfo;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.model.ids.GraphIdInfo;
import ru.yandex.crypta.graph.engine.proto.TGraph;
import ru.yandex.crypta.graph.engine.proto.TIdsInfo;
import ru.yandex.crypta.graph.engine.proto.TScore;
import ru.yandex.crypta.graph.engine.proto.TStats;
import ru.yandex.crypta.graph.engine.proto.TStatsOptions;
import ru.yandex.crypta.graph.engine.score.stats.EngineHelper;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph2.model.matching.score.MetricsTree;
import ru.yandex.crypta.lib.proto.identifiers.TGenericID;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

public class GraphEngineConverter {

    public GraphComponentWithInfo convert(TGraph graph) {

        List<Vertex> vertices = graph.getVerticesList()
                .stream()
                .map(v -> {
                    Identifier identifier = Identifier.fromProto(v.toByteArray());
                    return new Vertex(
                            identifier.getValue(),
                            Soup.CONFIG.name(identifier.getType())
                    );
                })
                .collect(toList());


        List<Edge> edges = graph.getEdgesList().stream().map(e ->
                new Edge(
                        vertices.get(e.getVertex1()),
                        vertices.get(e.getVertex2()),
                        Soup.CONFIG.name(e.getSourceType()),
                        Soup.CONFIG.name(e.getLogSource()),
                        e.getSurvivalWeight(),
                        List.of()
                )
        ).collect(toList());

        String cryptaId = Long.toUnsignedString(graph.getId());
        GraphComponent graphComponent = new GraphComponent(
                cryptaId,
                vertices,
                edges
        );
        GraphComponentWithInfo withInfo = new GraphComponentWithInfo(graphComponent);
        withInfo.setComponentsInfo(
                new GraphComponentInfo(
                        cryptaId,
                        new GraphId("", ""),
                        vertices.size(),
                        0,
                        calculateMetricsTree(graph)
//                        new MetricsTree(0)
                )
        );

        TIdsInfo protoIdsInfo = graph.getIdsInfo();
        List<GraphIdInfo> idsInfo = parseIdsInfo(
                Stream.of(
                        protoIdsInfo.getAppsInfoList().stream().map(x -> new RawProtoIdInfo(x.getId(), x)),
                        protoIdsInfo.getBrowsersInfoList().stream().map(x -> new RawProtoIdInfo(x.getId(), x)),
                        protoIdsInfo.getDevicesInfoList().stream().map(x -> new RawProtoIdInfo(x.getId(), x)),
                        protoIdsInfo.getSocdemInfoList().stream().map(x -> new RawProtoIdInfo(x.getId(), x))
                ).flatMap(x -> x)
        );

        withInfo.setIdsInfo(idsInfo);

        return withInfo;

    }

    private static class RawProtoIdInfo {
        TGenericID id;
        MessageOrBuilder info;

        RawProtoIdInfo(TGenericID id, MessageOrBuilder info) {
            this.id = id;
            this.info = info;
        }
    }

    private ArrayList<GraphIdInfo> parseIdsInfo(Stream<RawProtoIdInfo> protoIdInfos) {
        ObjectMapper jsonParser = new ObjectMapper();
        JsonFormat.Printer jsonPrinter = JsonFormat.printer();

        List<GraphIdInfo> rawIdInfos = protoIdInfos.map(protoInfo -> {
            Identifier id = Identifier.fromProto(protoInfo.id.toByteArray());
            GraphId graphId = new GraphId(
                    id.getValue(),
                    Soup.CONFIG.name(id.getType())
            );

            JsonNode jsonInfo;
            try {
                jsonInfo = jsonParser.readTree(jsonPrinter.print(protoInfo.info));
            } catch (IOException e) {
                throw new IllegalArgumentException(e);
            }

            Map<String, JsonNode> jsonProps = JsonUtils.jsonToMap(jsonInfo, r -> r);
            return new GraphIdInfo(graphId, jsonProps);

        }).collect(toList());

        HashMap<GraphId, GraphIdInfo> mergedInfos = new HashMap<>();
        for (GraphIdInfo graphIdInfo : rawIdInfos) {
            mergedInfos.merge(graphIdInfo.getId(), graphIdInfo, (a, b) -> a.addInfoRec(b.getInfo()));
        }

        return new ArrayList<>(mergedInfos.values());
    }

    private MetricsTree calculateMetricsTree(TGraph tGraph) {
        TStatsOptions graphEngineOptions = TStatsOptions.newBuilder().setCrossDeviceWeight(7.5).build();
        TStats stats = EngineHelper.collectProdStats(tGraph, graphEngineOptions);

        var children = stats
                .getScoresList()
                .stream()
                .collect(toMap(TScore::getName, TScore::getValue));

        return new MetricsTree(stats.getGeneralScore(), Cf.wrap(children));
    }
}
