package ru.yandex.crypta.graph.api.service.ids;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.ImmutableMap;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.api.proto.TSocdemProfile;
import ru.yandex.crypta.audience.proto.TAge;
import ru.yandex.crypta.audience.proto.TGender;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.settings.YtHumanMatchingGraphSettings;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;
import ru.yandex.misc.lang.number.UnsignedLong;

import static ru.yandex.bolts.collection.Tuple2.tuple;

public class YtIdsStorageService implements IdsStorageService {

    public static final ImmutableMap<String, TGender> GENDER_MAPPING = ImmutableMap.<String, TGender>builder()
            .put("m", TGender.MALE)
            .put("f", TGender.FEMALE)
            .build();
    public static final ImmutableMap<String, TAge> AGE_MAPPING = ImmutableMap.<String, TAge>builder()
            .put("0_17", TAge.FROM_0_TO_17)
            .put("18_24", TAge.FROM_18_TO_24)
            .put("25_34", TAge.FROM_25_TO_34)
            .put("35_44", TAge.FROM_35_TO_44)
            .put("45_54", TAge.FROM_45_TO_54)
            .put("55_99", TAge.FROM_55_TO_99)
            .build();
    private static final YPath IDS_STORAGE_BASE_PATH = YPath.simple("//home/crypta/production/ids_storage");
    private static final YPath YUID_WITH_ALL_PATH =
            YPath.simple("//home/crypta/production/state/graph/dicts/yuid_with_all");
    private static final YPath CRYPTA_ID_PROFILES = YPath.simple("//home/crypta/production/profiles/stages/cryptaid");

    private static final MapF<EIdType, List<String>> ID_INFO_TABLES_MAP = Cf.toMap(Cf.list(
            tuple(EIdType.YANDEXUID, Cf.list("yandexuid/yuid_with_all_info", "yandexuid/beh-profile")),
            tuple(EIdType.IDFA, Cf.list("device_id/app_metrica_month")),
            tuple(EIdType.GAID, Cf.list("device_id/app_metrica_month")),
            tuple(EIdType.UUID, Cf.list("uuid/app_metrica_month")),
            tuple(EIdType.EMAIL, Cf.list("email/emails_eu", "email/email_org_classification1"))
    ));

    private static final MapF<EIdType, String> ETERNAL_ID_INFO_TABLES_MAP = Cf.toMap(Cf.list(
            tuple(EIdType.YANDEXUID, "yandexuid/eternal"),
            tuple(EIdType.ICOOKIE, "icookie/eternal"),
            tuple(EIdType.MM_DEVICE_ID, "mm_device_id/eternal"),
            tuple(EIdType.IDFA, "idfa/eternal"),
            tuple(EIdType.GAID, "gaid/eternal"),
            tuple(EIdType.UUID, "uuid/eternal")
    ));

    private YtService yt;
    private YtHumanMatchingGraphSettings graphSettings;

    @Inject
    public YtIdsStorageService(YtService yt, YtHumanMatchingGraphSettings graphSettings) {
        this.yt = yt;
        this.graphSettings = graphSettings;
    }

    @Override
    public Optional<JsonNode> getIdInfo(GraphId id) {

        EIdType idType = Soup.CONFIG.getIdType(id.getIdType()).getType();
        ObjectNode result = JsonNodeFactory.instance.objectNode();

        if (ID_INFO_TABLES_MAP.containsKeyTs(idType)) {

            for (String tableName : ID_INFO_TABLES_MAP.getTs(idType)) {
                YPath tablePath = IDS_STORAGE_BASE_PATH.child(tableName);
                YPath pathByKey = tablePath.withExact(YtReadingUtils.exact(id.toKey()));

                Option<JsonNode> info = yt.readTableJson(pathByKey, r -> r).firstO();
                if (info.isPresent()) {
                    result.setAll((ObjectNode) info.get());
                }
            }
        }

        if (result.size() != 0) {
            return Optional.of(result);
        } else {
            return getEternalIdInfo(id);
        }

    }

    public Optional<JsonNode> getEternalIdInfo(GraphId id) {
        EIdType idType = Soup.CONFIG.getIdType(id.getIdType()).getType();

        if (ETERNAL_ID_INFO_TABLES_MAP.containsKeyTs(idType)) {
            String tableName = ETERNAL_ID_INFO_TABLES_MAP.getTs(idType);
            YPath tablePath = IDS_STORAGE_BASE_PATH.child(tableName);
            YPath pathByKey = tablePath.withExact(YtReadingUtils.exact(id.toIdTypeKey()));

            return yt.readTableJson(pathByKey, r -> r).firstO().toOptional();
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<JsonNode> getYuidAllInfo(GraphId yuid) {
        if (!yuid.getIdType().equals(GraphId.YANDEXUID_TYPE)) {
            throw new IllegalStateException("Only YANDEXUID is supported in this service");
        }

        YPath pathByKey = YUID_WITH_ALL_PATH.withExact(YtReadingUtils.exact(yuid.getIdValue()));

        return yt.readTableJson(pathByKey, r -> r).firstO().toOptional();

    }

    @Override
    public Optional<TSocdemProfile> getAggregatedCryptaIdSocdem(String cryptaId) {
        Optional<YPath> latest = yt.getHahn()
                .cypress()
                .list(CRYPTA_ID_PROFILES)
                .stream()
                .map(YTreeStringNode::getValue)
                .map(CRYPTA_ID_PROFILES::child)
                .sorted(Comparator.comparing(YPath::name))
                .findFirst();
        if (latest.isEmpty()) {
            return Optional.empty();
        }
        YPath pathByKey =
                latest.get().withExact(YtReadingUtils.exact(UnsignedLong.valueOf(Long.parseUnsignedLong(cryptaId))));
        return yt.readTableYson(pathByKey, this::extractAggregatedSocdem).firstO().toOptional();
    }

    @Override
    public Optional<TSocdemProfile> getGraphCryptaIdSocdem(String cryptaId, String matchingType) {
        YPath pathByKey = graphSettings.getPaths(matchingType).getVerticesPropertiesPath(cryptaId);
        return yt.readTableYson(pathByKey, this::extractGraphSocdem).firstO().toOptional();
    }

    private TSocdemProfile extractGraphSocdem(YTreeMapNode entry) {
        TSocdemProfile.Builder profile = TSocdemProfile.newBuilder();

        entry.getStringO("gender").map(GENDER_MAPPING::get).ifPresent(profile::setGenderExact);

        return profile.build();
    }

    private TSocdemProfile extractAggregatedSocdem(YTreeMapNode entry) {
        TSocdemProfile.Builder profile = TSocdemProfile.newBuilder();

        Map<String, YTreeNode> exactSocdem = entry.getOrThrow("exact_socdem").asMap();

        Optional.ofNullable(exactSocdem.get("age_segment"))
                .map(YTreeNode::stringValue)
                .map(AGE_MAPPING::get)
                .ifPresent(profile::setAgeExact);
        Optional.ofNullable(exactSocdem.get("gender"))
                .map(YTreeNode::stringValue)
                .map(GENDER_MAPPING::get)
                .ifPresent(profile::setGenderExact);

        entry.getOrThrow("all_age_segment_scores").asMap().forEach((k, v) -> profile
                .addAgeScoresBuilder()
                .setAge(AGE_MAPPING.get(k))
                .setScore(v.floatValue()));
        entry.getOrThrow("all_gender_scores").asMap().forEach((k, v) -> profile
                .addGenderScoresBuilder()
                .setGender(GENDER_MAPPING.get(k))
                .setScore(v.floatValue())
        );

        return profile.build();
    }


}
