package ru.yandex.crypta.graph.api.service.settings;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.graph.api.service.transformer.GraphShrinkingTransformer;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.lib.proto.TYtConfig;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.lang.Validate;

import static ru.yandex.crypta.graph.api.service.TransformerGraphService.MATCH_SCOPE_KEEP_SOUPY_INDEVICE;
import static ru.yandex.crypta.graph.api.service.TransformerGraphService.MATCH_SCOPE_SPANNING;

public class VultureGraphSettings implements GraphSettings {
    private static final String MATCH_TYPE_VULTURE = "vulture";
    private static final String MATCH_TYPE_EXP_VULTURE = "exp_vulture";

    private final static YPath BIGB_ID_TABLES_HOME = YPath.simple("//home/bigb/production");
    private final static YPath EXP_BIGB_ID_TABLES_HOME = YPath.simple("//home/bigb/prestable");

    private static final MapF<String, YPath> ID_TO_CRYPTA_ID = Cf.map(
            MATCH_TYPE_VULTURE, BIGB_ID_TABLES_HOME.child("IdToCryptaId"),
            MATCH_TYPE_EXP_VULTURE, EXP_BIGB_ID_TABLES_HOME.child("IdToCryptaId")
    );
    private static final MapF<String, YPath> CRYPTA_ID_TO_GRAPH = Cf.map(
            MATCH_TYPE_VULTURE, BIGB_ID_TABLES_HOME.child("CryptaIdToGraph"),
            MATCH_TYPE_EXP_VULTURE, EXP_BIGB_ID_TABLES_HOME.child("CryptaIdToGraph")
    );

    private static final String MATCH_SCOPE_CRYPTA_ID = "crypta_id";

    private static final List<String> MATCH_TYPES = Cf.list(
            MATCH_TYPE_VULTURE,
            MATCH_TYPE_EXP_VULTURE
    );

    private static final List<String> SCOPES = Stream.concat(
            List.of(
                    MATCH_SCOPE_CRYPTA_ID,
                    MATCH_SCOPE_SPANNING,
                    MATCH_SCOPE_KEEP_SOUPY_INDEVICE
            ).stream(),
            GraphShrinkingTransformer.MATCH_SCOPES.stream()
    ).collect(Collectors.toList());

    public static final List<String> ID_TYPES = Stream.concat(
            Stream.of(Soup.CONFIG.name(EIdType.CRYPTA_ID)),
            YtSoupGraphSettings.ID_TYPES.stream()
    ).collect(Collectors.toList());

    private final long rpcLookupTimeoutMilliseconds;

    @Inject
    public VultureGraphSettings(TYtConfig ytConfig) {
        Validate.notEmpty(ytConfig.getSenecaHosts());
        this.rpcLookupTimeoutMilliseconds = ytConfig.getRpcLookupTimeoutMilliseconds();
    }

    @Override
    public List<String> getSupportedMatchTypes() {
        return MATCH_TYPES;
    }

    @Override
    public List<String> getSupportedScopes() {
        return SCOPES;
    }

    @Override
    public List<String> getSupportedIdTypes() {
        return ID_TYPES;
    }

    public YPath getIdToCryptaIdTable(String matchType) {
        matchType = Objects.requireNonNullElse(matchType, MATCH_TYPE_VULTURE);

        return ID_TO_CRYPTA_ID.getO(matchType).getOrThrow(() ->
                Exceptions.illegal("MatchType should be one of " + getSupportedMatchTypes())
        );
    }

    public YPath getCryptaIdToGraph(String matchType) {
        matchType = Objects.requireNonNullElse(matchType, MATCH_TYPE_VULTURE);

        return CRYPTA_ID_TO_GRAPH.getO(matchType).getOrThrow(() ->
                Exceptions.illegal("MatchType should be one of " + getSupportedMatchTypes())
        );
    }

    public long getLookupTimeoutMilliseconds() {
        return this.rpcLookupTimeoutMilliseconds;
    }
}
