package ru.yandex.crypta.graph.api.service.settings;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.transformer.GraphShrinkingTransformer;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static ru.yandex.crypta.graph.api.service.TransformerGraphService.MATCH_SCOPE_KEEP_SOUPY_INDEVICE;
import static ru.yandex.crypta.graph.api.service.TransformerGraphService.MATCH_SCOPE_SPANNING;
import static ru.yandex.crypta.graph.api.service.TransformerGraphService.MATCH_SCOPE_SPANNING_ACTIVE;
import static ru.yandex.crypta.graph.api.service.TransformerGraphService.MATCH_SCOPE_SPANNING_ACTIVE_2;

public class YtHumanMatchingGraphSettings implements GraphSettings {

    public static final String MATCH_TYPE_YT_V2_PROD = "v2_prod_yt";
    public static final String MATCH_TYPE_YT_V2_EXP = "v2_exp_yt";
    public static final String MATCH_TYPE_YT_V2_CUSTOM = "v2_custom_path";

    public static final String MATCH_SCOPE_NEIGHBOURS = "neighbours";
    public static final String MATCH_SCOPE_CRYPTA_ID = "crypta_id";


    private static final List<String> SCOPES = Stream.concat(
            List.of(
                    MATCH_SCOPE_CRYPTA_ID,
                    MATCH_SCOPE_NEIGHBOURS,
                    MATCH_SCOPE_SPANNING,
                    MATCH_SCOPE_SPANNING_ACTIVE,
                    MATCH_SCOPE_SPANNING_ACTIVE_2,
                    MATCH_SCOPE_KEEP_SOUPY_INDEVICE
            ).stream(),
            GraphShrinkingTransformer.MATCH_SCOPES.stream()
    ).collect(Collectors.toList());

    public static final List<String> ID_TYPES = Stream.concat(
            Stream.of(Soup.CONFIG.name(EIdType.CRYPTA_ID)),
            YtSoupGraphSettings.ID_TYPES.stream()
    ).collect(Collectors.toList());


    private final MapF<String, GraphPaths> pathsByType;

    public YtHumanMatchingGraphSettings() {
        pathsByType = Cf.wrap(new LinkedHashMap<>());
        pathsByType.put(
                MATCH_TYPE_YT_V2_PROD,
                new GraphPaths("//home/crypta/production/state/graph/v2/matching")
        );
        pathsByType.put(
                MATCH_TYPE_YT_V2_EXP,
                new GraphPaths("//home/crypta/production/state/graph/v2exp/matching/workdir/output")
        );
    }

    @Override
    public List<String> getSupportedMatchTypes() {
        return Cf.list(
                MATCH_TYPE_YT_V2_PROD,
                MATCH_TYPE_YT_V2_EXP,
                MATCH_TYPE_YT_V2_CUSTOM
        );
    }

    @Override
    public boolean isSupportedMatchType(String matchType) {
        // support custom ypath as well
        return isValidYPath(matchType) || GraphSettings.super.isSupportedMatchType(matchType);
    }

    private boolean isValidYPath(String path) {
        try {
            YPath.simple(path);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    public GraphPaths getPaths(String matchType) {
        matchType = Objects.requireNonNullElse(matchType, MATCH_TYPE_YT_V2_PROD);
        if (isValidYPath(matchType)) {
            return new GraphPaths(YPath.simple(matchType));
        } else {
            return pathsByType.getO(matchType).getOrThrow(() ->
                    Exceptions.illegal("MatchType should be one of " + getSupportedMatchTypes() + " or valid YPath")
            );
        }
    }

    @Override
    public List<String> getSupportedScopes() {
        return SCOPES;
    }

    @Override
    public List<String> getSupportedIdTypes() {
        return ID_TYPES;
    }


    public static class GraphPaths {

        private YPath basePath;

        GraphPaths(YPath basePath) {
            this.basePath = basePath;
        }

        GraphPaths(String absolutePath) {
            basePath = YPath.simple(absolutePath);
        }

        public YPath getVerticesPath(GraphId graphId) {
            return basePath.child("vertices_no_multi_profile")
                    .withExact(YtReadingUtils.exact(graphId.toKey()));
        }

        public YPath getVerticesByCryptaIdPath(String cryptaId) {
            return basePath.child("vertices_no_multi_profile_by_crypta_id")
                    .withExact(YtReadingUtils.exact(cryptaId));
        }

        public YPath getEdgesPath(String cryptaId) {
            return basePath.child("edges_by_crypta_id")
                    .withExact(YtReadingUtils.exact(cryptaId));
        }

        public YPath getGraphStatsPath(String cryptaId) {
            return basePath.child("crypta_components_stats")
                    .withExact(YtReadingUtils.exact(cryptaId));
        }

        public YPath getGraphNeighboursPath(String cryptaId) {
            return basePath.child("crypta_components_neighbours")
                    .withExact(YtReadingUtils.exact(cryptaId));
        }

        public YPath getEdgesBetweenPath(String cryptaId) {
            return basePath.child("edges_between_components_by_crypta_id")
                    .withExact(YtReadingUtils.exact(cryptaId));
        }

        public YPath getVerticesPropertiesPath(String cryptaId) {
            return basePath.child("vertices_properties_by_crypta_id")
                    .withExact(YtReadingUtils.exact(cryptaId));
        }

        public YPath getMergeOffersPath(String cryptaId) {
            return basePath.child("merge_offers")
                    .withExact(YtReadingUtils.exact(cryptaId));
        }

        public YPath getSplitInfoPath(String cryptaId) {
            return basePath.child("split_info")
                    .withExact(YtReadingUtils.exact(cryptaId));
        }


        public YPath getIndevicePath(GraphId graphId) {
            return basePath.child("indevice_by_id_and_id_type")
                    .withExact(YtReadingUtils.exact(graphId.toKey()));
        }

    }
}
