package ru.yandex.crypta.graph.api.service.transformer;

import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import org.jgrapht.graph.WeightedMultigraph;

import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.service.YtSoupGraphService;
import ru.yandex.crypta.graph.api.service.settings.YtSoupGraphSettings;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

public class ExtendWithSoupEdgesTransformer implements GraphTransformer {

    private final YtService yt;
    private final boolean onlyInnerEdges;

    public ExtendWithSoupEdgesTransformer(YtService yt, boolean onlyInnerEdges) {
        this.yt = yt;
        this.onlyInnerEdges = onlyInnerEdges;
    }


    @Override
    public WeightedMultigraph<Vertex, Edge> transform(WeightedMultigraph<Vertex, Edge> graph) {

        Set<Vertex> ids = graph.vertexSet();

        List<CompletableFuture<Stream<Edge>>> requestsOfSoupRecs = ids
                .stream()
                .map(id -> selectEdges(id).thenApply(this::toEdges))
                .collect(toList());

        Set<Edge> soupEdges = requestsOfSoupRecs
                .stream()
                .map(CompletableFuture::join)
                .flatMap(stream -> stream)
                .collect(toSet());

        if (onlyInnerEdges) {
            soupEdges = soupEdges
                    .stream()
                    .filter(edge ->
                            ids.contains(edge.getVertex1()) && ids.contains(edge.getVertex2())
                    ).collect(toSet());
        }

        return JGraphTHelper.toSimpleGraph(soupEdges);
    }

    private CompletableFuture<UnversionedRowset> selectEdges(Vertex id) {
        return yt.getHahnRpc().selectRows(
                String.format("* FROM [%s] WHERE id1 = '%s' AND id1Type = '%s' LIMIT %d",
                        YtSoupGraphSettings.TABLE, id.getIdValue(), id.getIdType(), 1000)
        );
    }

    private Stream<Edge> toEdges(UnversionedRowset rowSet) {
        List<YTreeMapNode> soupRecs = rowSet.getYTreeRows(true);
        return soupRecs.stream().map(YtSoupGraphService::mapSoupRecToEdge);
    }

}
