package ru.yandex.crypta.graph.api.service.transformer;

import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.jgrapht.graph.WeightedMultigraph;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.Vertex;

public class SpanningActiveTransformer implements GraphTransformer {

    private static final SetF<String> ALLOWED_ID_TYPES = Cf.set(
            "yandexuid", "crypta_id1",
            "xuniq_guid", "device_id",
            "idfa", "gaid", "mm_device_id",
            "puid", "mac", "uuid"
    );
    private static final int ACTIVE_VERTEX_LIFETIME = 30;

    private Set<Vertex> activeVertices;
    private Set<Edge> requiredEdges;

    public SpanningActiveTransformer(Graph graph, boolean keepDirectEdges) {
        this.activeVertices = getActiveVertices(graph);

        if (keepDirectEdges) {
            this.requiredEdges = graph.getAllEdges()
                    .stream()
                    .filter(edge -> activeVertices.contains(edge.getVertex1()) && activeVertices
                            .contains(edge.getVertex2()))
                    .collect(Collectors.toSet());
        } else {
            this.requiredEdges = graph.getAllEdges()
                    .stream()
                    .filter(edge -> edge.getAttributes().contains("indevice"))
                    .collect(Collectors.toSet());
        }

    }

    @Override
    public WeightedMultigraph<Vertex, Edge> transform(WeightedMultigraph<Vertex, Edge> graph) {
        SpanningTreeTransformer spt = new SpanningTreeTransformer(activeVertices, requiredEdges);
        return spt.transform(graph);
    }

    private Set<Vertex> getActiveVertices(Graph graph) {

        // active by idsInfo
        Set<Vertex> activeVertices = graph.getIdsInfo().stream().map(idInfo ->
                new Vertex(idInfo.getId().getIdValue(), idInfo.getId().getIdType())
        ).collect(Collectors.toSet());

        activeVertices = activeVertices.stream().filter(v -> ALLOWED_ID_TYPES.containsTs(v.getIdType()))
                .collect(Collectors.toSet());

        // active by edges
        for (Edge edge : graph.getAllEdges()) {
            Optional<String> latestDate = edge.getDates().stream().max(String::compareTo);
            if (latestDate.isPresent()) {
                long daysGone = ChronoUnit.DAYS.between(LocalDate.parse(latestDate.get()), LocalDate.now());
                if (daysGone < ACTIVE_VERTEX_LIFETIME) {
                    activeVertices.add(edge.getVertex1());
                    activeVertices.add(edge.getVertex2());
                }
            }
        }
        return activeVertices;
    }
}
