package ru.yandex.crypta.graph.api.utils;

import java.io.ByteArrayOutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jgrapht.graph.Multigraph;
import org.jgrapht.io.Attribute;
import org.jgrapht.io.AttributeType;
import org.jgrapht.io.ComponentAttributeProvider;
import org.jgrapht.io.ComponentNameProvider;
import org.jgrapht.io.DOTExporter;
import org.jgrapht.io.DefaultAttribute;
import org.jgrapht.io.ExportException;
import org.jgrapht.io.IntegerComponentNameProvider;

import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.Vertex;


public class GraphvizExporter {
    private GraphvizExporter() {
    }

    public static String exportGraphToDot(Graph graph, boolean needColor) throws ExportException {
        ByteArrayOutputStream os = new ByteArrayOutputStream();

        List<Vertex> vertices = new ArrayList<>();
        List<Edge> edges = new ArrayList<>();

        graph.getGraphComponents().forEach(component ->
        {
            vertices.addAll(component.getVertices());
            edges.addAll(component.getEdges());
        });

        GraphvizExporter.toDot(vertices, edges, os, needColor);

        return os.toString(StandardCharsets.UTF_8);
    }


    @SuppressWarnings({"unchecked", "rawtypes"})
    private static <V extends ExportableVertex, E extends ExportableEdge> void toDot(
            List<V> vertices, List<E> edges, ByteArrayOutputStream os, Boolean needColor
    ) throws ExportException {

        Multigraph graph = new Multigraph(ExportableVertex.class);

        IntegerComponentNameProvider<V> vertexIdProvider = new IntegerComponentNameProvider();
        ComponentNameProvider<V> vertexLabelProvider = ExportableVertex::getFullId;
        ComponentNameProvider<E> edgeLabelProvider = e -> e.getSourceType() + "_" + e.getLogSource();

        DOTExporter export;
        ComponentAttributeProvider<V> vertexAttributeProvider;
        ComponentAttributeProvider<E> edgeAttributeProvider;

        if (needColor) {
            vertexAttributeProvider = vert -> {
                Map<String, Attribute> map = new HashMap<>();
                map.put("color", new DefaultAttribute<>(vert.getColor(), AttributeType.STRING));
                return map;
            };

            edgeAttributeProvider = edge -> {
                Map<String, Attribute> map = new HashMap<>();
                map.put("color", new DefaultAttribute<>(edge.getColor(), AttributeType.STRING));
                return map;
            };
        } else {
            vertexAttributeProvider = vert -> new HashMap<>();
            edgeAttributeProvider = edge -> new HashMap<>();
        }

        export = new DOTExporter<>(vertexIdProvider, vertexLabelProvider, edgeLabelProvider,
                vertexAttributeProvider, edgeAttributeProvider);

        Map<String, V> vertexMap = new HashMap<>();
        vertices.forEach(vertex -> vertexMap.put(vertex.getIdValue() + " " + vertex.getIdType(), vertex));

        vertices.forEach(graph::addVertex);

        edges.forEach(edge -> graph.addEdge(
                vertexMap.get(edge.getId1() + " " + edge.getId1Type()),
                vertexMap.get(edge.getId2() + " " + edge.getId2Type()),
                edge
        ));

        export.exportGraph(graph, os);
    }
}
