package ru.yandex.crypta.graph.api.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.commons.codec.digest.DigestUtils;

import ru.yandex.crypta.graph.api.model.graph.Edge;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.GraphComponent;
import ru.yandex.crypta.graph.api.model.graph.GraphComponentInfo;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.model.ids.GraphIdInfo;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph.soup.config.proto.ESourceType;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.proto.identifiers.ERepresentedObjectType;
import ru.yandex.crypta.lib.proto.identifiers.TIdType;

public class SecureGraphHelper {

    public static final String EXTERNAL_PARTNER_MASKED_TYPE = "external_partner";
    public static final String MASKED_TYPE = Soup.CONFIG.name(ESourceType.MASKED_TYPE);

    public Graph createMaskedGraph(Graph graph) {
        List<GraphComponent> newComponents = new ArrayList<>();

        for (GraphComponent component : graph.getGraphComponents()) {

            List<Vertex> maskedVertices = component.getVertices()
                    .stream()
                    .map(this::createMaskedVertex)
                    .collect(Collectors.toList());
            List<Edge> maskedEdges = component.getEdges()
                    .stream()
                    .map(this::createMaskedEdge)
                    .collect(Collectors.toList());

            GraphComponent newComponent = new GraphComponent(component.getCryptaId(), maskedVertices, maskedEdges);
            newComponents.add(newComponent);
        }

        List<Edge> maskedEdgesBetween = graph.getEdgesBetweenComponents()
                .stream()
                .map(this::createMaskedEdge)
                .collect(Collectors.toList());

        List<GraphIdInfo> maskedIdsInfo = graph.getIdsInfo()
                .stream()
                .map(info -> new GraphIdInfo(
                        maskedGraphId(info.getId()),
                        info.getInfo()
                )).collect(Collectors.toList());

        Map<String, GraphComponentInfo> maskedComponentsInfo = graph.getComponentsInfo()
                .entrySet()
                .stream()
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        e -> maskedComponentsInfo(e.getValue())
                ));

        Graph maskedGraph = new Graph(newComponents);
        maskedGraph.setEdgesBetweenComponents(maskedEdgesBetween);
        maskedGraph.setIdsInfo(maskedIdsInfo);
        maskedGraph.setComponentsInfo(maskedComponentsInfo);
        maskedGraph.setMergeInfo(graph.getMergeInfo());

        return maskedGraph;
    }

    private GraphId maskedGraphId(GraphId id) {
        return GraphId.fromVertex(createMaskedVertex(id.toVertex()));
    }

    private GraphComponentInfo maskedComponentsInfo(GraphComponentInfo info) {
        return new GraphComponentInfo(
                info.getCryptaId(),
                maskedGraphId(info.getMainId()),
                info.getVerticesCount(),
                info.getNeighboursCount(),
                info.getMetricsTree()
        );
    }

    public static boolean isPersonalId(String idType) {
        TIdType enumIdType = Soup.CONFIG.getIdType(idType);
        return enumIdType == null || ( // unknown id is personal by default for security reasons
                enumIdType.getType() != EIdType.CRYPTA_ID
                        && enumIdType.getType() != EIdType.PUID
                        && enumIdType.getRepObj().equals(ERepresentedObjectType.PERSON)
        );
    }

    public Vertex createMaskedVertex(Vertex vertex) {
        String idValue = vertex.getIdValue();
        String idType = vertex.getIdType();
        TIdType enumIdType = Soup.CONFIG.getIdType(idType);

        if (!isPersonalId(idType)) {
            return new Vertex(idValue, idType);
        }

        String idHash = DigestUtils.md5Hex(idValue);
        String maskedIdValue = DigestUtils.md5Hex(idHash);

        String maskedIdType;

        if (enumIdType != null && (Objects.equals(enumIdType.getType(), EIdType.DIT_ID)
                || Objects.equals(enumIdType.getType(), EIdType.AVITO_HASH))) {
            maskedIdType = EXTERNAL_PARTNER_MASKED_TYPE; // to hide dit and avito partners
        } else {
            maskedIdType = idType;
        }

        return new Vertex(maskedIdValue, maskedIdType);
    }

    public Edge createMaskedEdge(Edge edge) {
        TIdType enumId1Type = Soup.CONFIG.getIdType(edge.getId1Type());
        TIdType enumId2Type = Soup.CONFIG.getIdType(edge.getId2Type());

        boolean id1IsNotPerson = (Objects.nonNull(enumId1Type) && !enumId1Type.getRepObj()
                .equals(ERepresentedObjectType.PERSON));
        boolean id2IsNotPerson = (Objects.nonNull(enumId2Type) && !enumId2Type.getRepObj()
                .equals(ERepresentedObjectType.PERSON));

        if (id1IsNotPerson && id2IsNotPerson) {
            return edge;
        }

        Vertex maskedVertex1 = createMaskedVertex(edge.getVertex1());
        Vertex maskedVertex2 = createMaskedVertex(edge.getVertex2());

        return new Edge(
                maskedVertex1,
                maskedVertex2,
                MASKED_TYPE,
                MASKED_TYPE,
                edge.getWeight(),
                edge.getDates()
        );
    }
}
