import logging
import sys
import datetime

import google.protobuf.text_format
from library.python import resource
from reactor_client.reactor_api import ReactorAPIClientV1
import reactor_client.reactor_objects as r_objects

from crypta.lib.python.getoptpb import ParsePbOptions
from crypta.lib.python.yt.yt_helpers import get_yt_client
import crypta.graph.archivator.proto.config_pb2 as proto_config
from crypta.graph.archivator.lib import BackupTask


def make_logger():
    handler = logging.StreamHandler(sys.stderr)
    handler.setLevel(logging.INFO)
    logging.root.addHandler(handler)
    logging.root.setLevel(logging.INFO)


def send_reaction(config, base_namespace, reactions, cluster):
    client = ReactorAPIClientV1(base_url=config.Url, token=config.Token)
    for (suffix, dtstr), tables in reactions.items():
        artifact_id = r_objects.ArtifactIdentifier(
            namespace_identifier=r_objects.NamespaceIdentifier(
                namespace_path="/".join((base_namespace, suffix))
            )
        )

        client.artifact_instance.instantiate(
            artifact_identifier=artifact_id,
            metadata=r_objects.Metadata(
                type_="/yandex.reactor.artifact.YtPathListArtifactValueProto",
                dict_obj={
                    "values": [{"cluster": cluster, "path": table} for table in tables]
                },
            ),
            user_time=datetime.datetime.strptime(dtstr, "%Y-%m-%d"),
        )


def actualize_source(source, yt_client):
    raw_source = source
    if raw_source.endswith("{last_ts}"):
        return max(
            yt_client.list(raw_source.split("{last_ts}")[0].rstrip("/"), absolute=True),
            key=lambda tbl: str(tbl),
        )
    return raw_source


def main():
    make_logger()

    proto = ParsePbOptions(proto_config.TArchivatorConfig)
    if not proto.Backupers:
        logging.info("Backupers are empty, use default")
        buff = google.protobuf.text_format.Parse(
            resource.find("/conf/default.pb.txt"), proto_config.TArchivatorConfig()
        )
        buff.MergeFrom(proto)
        proto = buff

    yt_client = get_yt_client(proto.Yt.Proxy, yt_pool=proto.Yt.Pool)
    yt_client.config["tablets_ready_timeout"] = proto.Yt.TabletsReadyTimeout
    task = BackupTask(yt_client)
    source = actualize_source(proto.Source, yt_client)
    task.run(source, proto.BackupDir, proto.Backupers)

    if proto.ReactorNamespace:
        send_reaction(
            proto.Reactor,
            proto.ReactorNamespace,
            task.reactions,
            cluster=proto.Yt.Proxy.split(".")[0],
        )
