#include <crypta/graph/bloom_email/lib/eu_emails_filter.h>
#include <crypta/graph/bloom_email/proto/builder.pb.h>

#include <crypta/lib/native/cmd_args/parse_pb_options.h>

#include <library/cpp/logger/global/global.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/operation.h>
#include <util/stream/file.h>
#include <util/stream/input.h>

using namespace NCrypta::NEmailsFilter;

namespace {
    class TBuilderHelper {
    public:
        TBuilderHelper() = default;
        void Add(const TString& email) {
            if (TEuEmailsFilter::HasEuTld(email)) {
                return;
            }

            Emails.push_back(email);
        }
        TEuEmailsFilter CreateFilter() {
            TEuEmailsFilter filter(Emails.size());
            for (const auto& email : Emails) {
                filter.AddEmail(email);
            }

            return filter;
        }
        uint64_t Size() {
            return Emails.size();
        }

    private:
        TVector<TString> Emails;
    };

    void ExportFilterToYt(const NYT::IClientBasePtr& client, const NYT::TYPath& to, const TEuEmailsFilter& filter) {
        if (!client->Exists(to)) {
            client->Create(to, NYT::ENodeType::NT_FILE, NYT::TCreateOptions().Recursive(true));
        }

        auto ytFileWriter = client->CreateFileWriter(to);
        filter.Save(&(*ytFileWriter));
        ytFileWriter->Flush();
    }

    void ExportFilterToLocal(const TString& to, const TEuEmailsFilter& filter) {
        TFileOutput outfile{to};
        filter.Save(&outfile);
    }
}

int main(int argc, const char* argv[]) {
    TBuilderHelper helper;

    auto config = NCrypta::ParsePbOptions<TBuilderOptions>(argc, argv);
    DoInitGlobalLog(CreateLogBackend("cerr", static_cast<ELogPriority>(config.GetLogLevel()), true));

    DEBUG_LOG << "Config successfully parsed, log initialized." << Endl;

    NYT::Initialize(argc, argv);
    auto client = NYT::CreateClient(config.GetYt().GetProxy());

    INFO_LOG << "Reading emails from: " << config.GetEmailsTablePath() << Endl;

    auto reader = client->CreateTableReader<TBuilderEmailEntry>(config.GetEmailsTablePath());
    for (; reader->IsValid(); reader->Next()) {
        auto row = reader->GetRow();

        helper.Add(row.GetEmail());
    }

    INFO_LOG << "Table had been read. Total emails to add to filter: " << helper.Size() << Endl;

    const TEuEmailsFilter& filter = helper.CreateFilter();

    INFO_LOG << "Filter had been built. Size: " << 1. * filter.GetBitCount() / (8 * 1024) << " Kb."
             << " Writing filter to: " << config.GetFilterOutputPath() << Endl;

    if (config.GetLocal()) {
        ExportFilterToLocal(config.GetFilterOutputPath(), filter);
    } else {
        ExportFilterToYt(client, config.GetFilterOutputPath(), filter);
    }

    DEBUG_LOG << "Done" << Endl;
}
