#include "eu_emails_filter.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/writer/json_value.h>
#include <util/generic/hash_set.h>
#include <util/stream/file.h>

namespace NCrypta::NEmailsFilter {
    // may be we can have a constructor with fixed size?
    TEuEmailsFilter::TEuEmailsFilter(size_t elemcount, float error)
        : BloomFilter(elemcount, error)
    {
    }

    bool TEuEmailsFilter::HasEuTld(TStringBuf email) {
        static const THashSet<TString> euPrefixes{
            "nl", "it", "cz", "dk", "pl", "fi", "lt", "lu", "fr", "se", "lv",
            "hr", "gi", "ie", "ch", "no", "sk", "bg", "ro", "pt", "ee", "mt",
            "is", "uk", "si", "at", "de", "gr", "be", "hu", "cy", "es"};
        return euPrefixes.contains(email.RNextTok("."));
    }

    void TEuEmailsFilter::AddEmail(const TString& email) {
        if (!HasEuTld(email)) {
            BloomFilter.Add(email);
        }
    }

    bool TEuEmailsFilter::IsEuEmail(TStringBuf email) const {
        return HasEuTld(email) || BloomFilter.Has(email);
    }

    size_t TEuEmailsFilter::GetBitCount() const {
        return BloomFilter.GetBitCount();
    }

    size_t TEuEmailsFilter::GetHashCount() const {
        return BloomFilter.GetHashCount();
    }

    void TEuEmailsFilter::Save(IOutputStream* out) const {
        BloomFilter.Save(out);
    }

    void TEuEmailsFilter::Load(IInputStream* inp) {
        BloomFilter.Load(inp);
    }

    TEuEmailsFilter CreateEuEmailsFilter(const TString& fileName) {
        TFileInput input{fileName};
        TString line;
        TVector<TString> emailsToAdd;

        while (input.ReadLine(line)) {
            NJson::TJsonValue jsonValue;
            NJson::ReadJsonTree(line, &jsonValue);

            auto email = jsonValue["email"].GetString();
            if (TEuEmailsFilter::HasEuTld(email)) {
                continue;
            }

            emailsToAdd.push_back(email);
        }

        TEuEmailsFilter filter(emailsToAdd.size());
        for (const auto& email : emailsToAdd) {
            filter.AddEmail(email);
        }

        return filter;
    }
}
