import bitarray
import datetime

from ads.bsyeti.libs.log_protos import crypta_profile_pb2
from crypta.lib.python.identifiers.generic_id import GenericID


class Keyword(object):
    """Base keyword class"""

    def __init__(self, _):
        pass

    def parse(self, timestamp, key, kw_value):
        self.proto.yandex_id = int(kw_value.get("yuid") or key)
        self.item.keyword_id = kw_value["keyword"]

        self.set_item(kw_value)

    def set_item(self, kw_value):
        raise NotImplementedError

    def serialized_proto(self, timestamp, key, kw_value):
        self.proto = crypta_profile_pb2.TCryptaLog()
        self.proto.timestamp = timestamp
        self.item = self.proto.items.add()

        self.parse(timestamp, key, kw_value)
        return self.proto.SerializeToString()


class DictKeyword(Keyword):
    """Base keyword class"""

    KW_ID = 0

    def parse(self, timestamp, key, kw_value):
        raise NotImplementedError

    def set_item(self, kw_value):
        raise NotImplementedError

    def serialized_proto(self, timestamp, key, kw_value):
        if self.KW_ID != kw_value["keyword"]:
            raise ValueError("Incorrect keyword {!r}".format(kw_value))

        if self._skip_upload(kw_value, timestamp):
            return None
        return super(DictKeyword, self).serialized_proto(timestamp, key, kw_value)

    def _skip_upload(self, kw_value, timestamp):
        return False


class RepeatedUint(Keyword):

    """Example: value=4220463190:14938493:2307066304:1969428879"""

    def set_item(self, tskv):
        self.item.uint_values[:] = [int(x) for x in tskv["value"].split(":")]


class Bytes(Keyword):

    """Example: value=010000110011011101001010001101100100100001..."""

    def set_item(self, tskv):
        """Convert value of 0..1 to b64 like string"""
        self.item.string_value = bitarray.bitarray(tskv["value"]).tobytes()


class CidHistory(DictKeyword):

    KW_ID = 1187

    def parse(self, timestamp, key, kw_value):
        if kw_value["id_type"] == "yandexuid":
            self.proto.yandex_id = int(kw_value["id"])
        elif kw_value["id_type"] == "cryptaid":
            self.proto.crypta_id = int(kw_value["id"])
        elif kw_value["id_type"] == "icookie":
            # crutch for icookie == yandexuid
            self.proto.yandex_id = int(kw_value["id"])
        elif kw_value["id_type"] == "duid":
            self.proto.domain_user_id = int(kw_value["id"])
        elif kw_value["id_type"] == "puid":
            self.proto.puid = int(kw_value["id"])
        elif kw_value["id_type"] == "idfa":
            self.proto.idfa = kw_value["id"]
        elif kw_value["id_type"] == "gaid":
            self.proto.gaid = kw_value["id"]
        elif kw_value["id_type"] == "oaid":
            self.proto.oaid = kw_value["id"]
        elif kw_value["id_type"] == "mm_device_id":
            self.proto.mm_devid = kw_value["id"]
        else:
            raise ValueError("Non packable id_type {!r}".format(kw_value))

        self.item.keyword_id = self.KW_ID
        self.item.uint_values[:] = [value for pair in kw_value["crypta_ids"] for value in pair]

    def _skip_upload(self, kw_value, timestamp, round_base=10, active_limit=3):
        now = datetime.datetime.fromtimestamp(timestamp)
        last_cid_update = datetime.datetime.fromtimestamp(kw_value["crypta_ids"][-1][0])

        return not any(
            (
                now.day % round_base == GenericID(kw_value["id_type"], kw_value["id"]).half % round_base,
                now - datetime.timedelta(days=active_limit) < last_cid_update,
            )
        )


class IsStaff(DictKeyword):

    KW_ID = 1140

    class KwValues:
        STAFF = 1
        USERDATA = 2

    def parse(self, timestamp, key, kw_value):
        if self.KW_ID != kw_value["keyword"]:
            raise ValueError("Incorrect keyword {!r}".format(kw_value))

        if kw_value["id_type"] == "puid":
            self.proto.puid = int(kw_value["id"])
        else:
            raise ValueError("Non packable id_type {!r}".format(kw_value))

        self.item.keyword_id = self.KW_ID

        if kw_value["from_staff"]:
            self.item.uint_values.append(self.KwValues.STAFF)
        if kw_value["from_userdata"]:
            self.item.uint_values.append(self.KwValues.USERDATA)


KEYWORDS_MAP = {
    353: Bytes("households"),
    723: RepeatedUint("additional.mobile.apps"),
    1187: CidHistory("crypta-cid-history"),
    1140: IsStaff("crypta-is-staff"),
}
