import base64
import logging

from ads.bsyeti.libs.events.proto.cleaner_pb2 import TProfileCleanerConfig

from crypta.graph.bochka.lib.keywords import KEYWORDS_MAP
from crypta.graph.engine.proto.info_pb2 import TIdsInfo, TBrowserInfo, TDeviceInfo, TAppInfo
from crypta.graph.rt.events.proto.soup_pb2 import TSoupEvent, ESource
from crypta.graph.soup.config.proto.bigb_pb2 import TLinks, EBbLinkUsage
from crypta.graph.soup.config.proto.edge_type_pb2 import TEdgeProps
from crypta.graph.vavilov.proto.vavilov_pb2 import TVavilov
from crypta.lib.python.identifiers.generic_id import GenericID
from crypta.lib.python.identifiers import yabs
from crypta.lib.python import tskv


from crypta.graph.soup.config.python import (  # N811 # noqa
    EDGE_TYPE as edges,
    ID_TYPE as id_type,
    LOG_SOURCE as log_source,
    SOURCE_TYPE as source_type,
    Edges,
)

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


def tskv_to_dict(value):
    value = tskv.tskv_to_dict(value)
    value["keyword"] = int(value["keyword"])
    return value


def tskv_to_serialized_proto(timestamp, row):
    key = row["key"]
    value = tskv_to_dict(row["value"])
    keyword = value["keyword"]
    return KEYWORDS_MAP[keyword].serialized_proto(timestamp, key, value)


def value_to_serialized_proto(timestamp, row):
    return KEYWORDS_MAP[row["keyword"]].serialized_proto(timestamp, row["id"], row)


def b64value(row):
    row["value"] = base64.b64encode(row["value"])
    return row


def row_to_soup_edge(timestamp, row):
    proto = TSoupEvent()
    try:
        proto.Edge.SourceType = source_type.by_ext(row["sourceType"]).Type
        proto.Edge.LogSource = log_source.by_ext(row["logSource"]).Type
    except KeyError:
        logger.exception("Invalid log source or source type")
        return None

    for idx in (1, 2):
        identifier = GenericID(id_type=row["id{idx}Type".format(idx=idx)], id_value=row["id{idx}".format(idx=idx)])
        if identifier.is_significant():
            getattr(proto.Edge, "Vertex{idx}".format(idx=idx)).CopyFrom(identifier.to_proto())
        else:
            logger.debug("Insignificant identifier %s at row %s", identifier, row)
            return None
    proto.Unixtime = timestamp
    proto.Source = ESource.EDGE
    return base64.b64encode(proto.SerializeToString())


def set_value(proto, key, value):
    if value is None:
        return
    try:
        # try to skip bad chars
        value = value.decode("utf-8", "ignore")
    except Exception:
        pass
    setattr(proto, key, value)


def _to_eternal_brower_info(row, identifier):
    proto = TBrowserInfo()
    proto.Id.CopyFrom(identifier)

    set_value(proto, "OsName", row["os_name"])
    set_value(proto, "OsVersion", row["os_version"])
    set_value(proto, "OsFamily", row["os_family"])

    set_value(proto, "BrowserName", row["browser_name"])
    set_value(proto, "BrowserVersion", row["browser_version"])

    set_value(proto, "DateBegin", row["date_begin"])
    set_value(proto, "DateEnd", row["date_end"])

    for key, value in row.iteritems():
        if not key.startswith("is_"):
            continue
        proto_key = "Is{key}".format(key=key.replace("is_", "").capitalize())
        setattr(proto, proto_key, value)

    return proto


def _to_eternal_device_info(row, identifier):
    proto = TDeviceInfo()
    proto.Id.CopyFrom(identifier)

    set_value(proto, "OsName", row["os"])
    set_value(proto, "OsVersion", row["os_version"])

    set_value(proto, "Model", row["model"])
    set_value(proto, "Manufacturer", row["manufacturer"])

    set_value(proto, "ScreenWidth", row["screen_width"])
    set_value(proto, "ScreenHeight", row["screen_height"])

    set_value(proto, "DateBegin", row["date_begin"])
    set_value(proto, "DateEnd", row["date_end"])
    return proto


def _to_eternal_app_info(row, identifier):
    proto = TAppInfo()
    proto.Id.CopyFrom(identifier)

    set_value(proto, "OsName", row["os"])

    set_value(proto, "AppId", row["app_id"])
    set_value(proto, "AppVersion", row["app_version"])

    set_value(proto, "DateBegin", row["date_begin"])
    set_value(proto, "DateEnd", row["date_end"])

    for key, value in (row["api_keys"] or {}).iteritems():
        proto.ApiKeys[key] = value
    return proto


def row_to_eternal_info(_, row):
    identifier = GenericID(id_type=row["id_type"], id_value=row["id"])
    if not identifier.is_significant():
        logger.debug("Insignificant identifier %s at row %s", identifier, row)
        return None

    identifier_type = id_type.by_type(identifier.type)
    container = TIdsInfo()
    if identifier_type in (id_type.YANDEXUID, id_type.ICOOKIE):
        info = container.BrowsersInfo.add()
        info.CopyFrom(_to_eternal_brower_info(row, identifier.to_proto()))
    elif identifier_type in (id_type.IDFA, id_type.GAID, id_type.MM_DEVICE_ID):
        info = container.DevicesInfo.add()
        info.CopyFrom(_to_eternal_device_info(row, identifier.to_proto()))
    elif identifier_type in (id_type.UUID,):
        info = container.AppsInfo.add()
        info.CopyFrom(_to_eternal_app_info(row, identifier.to_proto()))
    else:
        logger.debug("Invalid identifier Type %s at row %s", identifier, row)
        return None
    return container.SerializeToString()


def row_to_vulture(timestamp, row, prod_enable=True, exp_enable=True):
    proto = TLinks()
    # check is bochka enabled
    try:
        source = source_type.by_ext(row["sourceType"])
        log = log_source.by_ext(row["logSource"])
        id1type = id_type.by_ext(row["id1Type"])
        id2type = id_type.by_ext(row["id2Type"])
        edge = edges.get_edge_type(id1type, id2type, source, log)
    except (ValueError, KeyError):
        logger.exception("Invalid edge")
        return None

    if not (edge.Usage.Bochka or edge.Usage.BochkaExp):
        logger.debug("Skip non bochka edge row %s", row)
        return None

    proto.SourceType = source.Type
    proto.LogSource = log.Type

    for index in (1, 2):
        identifier = GenericID(
            id_type=row["id{index}Type".format(index=index)], id_value=row["id{index}".format(index=index)]
        )
        if identifier.is_significant():
            vertex = proto.Vertices.add()
            vertex.IdType = identifier.type
            vertex.Id = identifier.normalize
        else:
            logger.debug("Insignificant identifier %s at row %s", identifier, row)
            return None

    proto.LogEventTimestamp = timestamp
    proto.Indevice = edge.Props.DeviceBounds == TEdgeProps.EDeviceBounds.INDEVICE

    # set proto usage for prod vulter only
    if prod_enable and edge.Usage.Bochka:
        proto.Usage.append(EBbLinkUsage.PROD)
    if exp_enable and edge.Usage.BochkaExp:
        proto.Usage.append(EBbLinkUsage.EXP)

    return proto.SerializeToString()


def row_to_vavilov(timestamp, generate_date, row):
    proto = TVavilov()

    proto.Id = row["id"]
    proto.IdType = row["id_type"]
    proto.CryptaId = row["crypta_id"]
    proto.IsDeleted = row["crypta_id"] == "0"

    proto.Unixtime = timestamp
    proto.GenerateDate = generate_date

    return base64.b64encode(proto.SerializeToString())


def row_to_profile_cleaner(timestamp, row, reason):
    gid = GenericID(id_type=row["id_type"], id_value=row["id"])
    profile_uniq = yabs.convert_to_storage_id(gid.type, gid.normalize)
    if profile_uniq is None:
        return None

    proto = TProfileCleanerConfig()
    proto.ProfileID = profile_uniq
    proto.TimeStamp = timestamp
    proto.NeedRemove = True
    proto.Reason = reason

    return proto.SerializeToString()
