import contextlib
import datetime

from crypta.graph.soup.config.python import LOG_SOURCE
from crypta.graph.data_import.stream.lib.tasks.base import StreamBaseYqlTask, Parameter

import crypta.lib.python.bt.conf.conf as conf


class AppMetrikaTask(StreamBaseYqlTask):

    """ Parse app metrika as bt task """

    log_source = LOG_SOURCE.METRIKA_MOBILE_LOG
    limit = Parameter(parse=int, default=190)  # 5h * (3 logs x 5 min + 1 logs x 30 min)
    filter_system_events = Parameter(
        parse=lambda value: value.lower() in {"+", "yes", "true", "t", "1"}, default="True"
    )

    def observed_paths(self):
        yield conf.paths.logfeller.metrika_mobile_log.stream
        yield conf.paths.logfeller.metrika_mobile_private_log.stream
        yield conf.paths.logfeller.metrika_mobile_yabro_log.stream
        yield conf.paths.logfeller.metrika_mobile_pp_log.stream

    @property
    def query_template(self):
        return "metrica.sql.j2"

    def get_context_data(self, **kwargs):
        context = super(AppMetrikaTask, self).get_context_data(**kwargs)

        def is_yabro_log(path):
            return path.startswith(conf.paths.logfeller.metrika_mobile_yabro_log.stream) \
                or path.startswith(conf.paths.logfeller.metrika_mobile_yabro_log.day)

        def is_pp_log(path):
            return path.startswith(conf.paths.logfeller.metrika_mobile_pp_log.stream) \
                or path.startswith(conf.paths.logfeller.metrika_mobile_pp_log.day)

        def is_appm_log(path):
            return not is_yabro_log(path) and not is_pp_log(path)

        context.update(
            metrica_input=[path for path in self.unprocessed_tables_final if is_appm_log(path)],
            yabro_input=[path for path in self.unprocessed_tables_final if is_yabro_log(path)],
            pp_input=[path for path in self.unprocessed_tables_final if is_pp_log(path)],
            cross_mobmet=conf.paths.storage.cross_mobmet,
            metrika_params_owners=conf.paths.storage.metrika_params_owners,
            filter_system_events=self.filter_system_events,
        )
        return context

    @contextlib.contextmanager
    def run_context(self):
        with super(AppMetrikaTask, self).run_context() as ctx:
            yield ctx
        self._set_expiration()

    def _set_expiration(self):
        self.yt.set(
            "{path}/{date:%Y-%m-%d}/@expiration_time".format(path=self.output_extra_data_dir, date=self.date),
            "{date:%Y-%m-%d} 12:00:00.0+00:00".format(date=self.date + datetime.timedelta(days=conf.proto.DaysTTL)),
        )
