import json
import urllib
from functools import partial
from urlparse import urlparse

from crypta.graph.soup.config.python import (  # N811 # noqa
    EDGE_TYPE as edges,
    ID_TYPE as ids,
    LOG_SOURCE as log_source,
    SOURCE_TYPE as source_type,
)

from crypta.graph.data_import.stream.lib.tasks.base import StreamBaseYqlTask
import crypta.lib.python.bt.conf.conf as conf


OOM_LIMIT = 100
VK = "vk"
OK = "ok"
EMAIL = "email"


def make_soup_rec(id1, id2, edge_type, dt):
    return {
        "id1": id1,
        "id1Type": edge_type.Id1Type.Name,
        "id2": id2,
        "id2Type": edge_type.Id2Type.Name,
        "sourceType": edge_type.SourceType.Name,
        "logSource": edge_type.LogSource.Name,
        "dates": [dt],
    }


def get_field_value(field_name, value, separator="\t", equal_sign="="):
    res = ""
    idx_start = value.find(separator + field_name + equal_sign)
    if idx_start != -1:
        idx_start = idx_start + len(field_name) + len(equal_sign) + len(separator)
    elif value.startswith(field_name + equal_sign):
        idx_start = len(field_name) + len(equal_sign)
    else:
        return res

    if idx_start >= len(value):
        return res
    idx_end = value.find(separator, idx_start)
    if idx_end > 0:
        res = value[idx_start:idx_end]
    else:
        res = value[idx_start:]

    return res


def map_bar_log(  # C901 # noqa
    rec,
    ajax_parsers,
    yuid_vkcom_barnavig,
    yuid_okru_barnavig,
    yuid_mailru_barnavig,
    yuid_r1_barnavig,
    yuid_ui_barnavig,
    yuid_uuid_barnavig,
    yuid_yuid_yabro_cookie_mining,
):
    yuid = rec.get("yandexuid")
    dt = rec.get("iso_eventtime").split(" ")[0]

    if yuid and yuid != "-":
        http_params = rec.get("http_params", "")
        orig_ui = get_field_value("ui", http_params, "&")
        r1 = get_field_value("r1", http_params, "&") or ""
        uuid = get_field_value("uuid", http_params, "&")

        # parse ajax requets
        decoded_ajax = get_field_value("decoded_ajax", http_params, "&")
        if decoded_ajax:
            try:
                for id_value, id_type in parse_ajax(decoded_ajax, ajax_parsers):
                    if id_type == VK:
                        edge_type = yuid_vkcom_barnavig
                    elif id_type == OK:
                        edge_type = yuid_okru_barnavig
                    elif id_type == EMAIL:
                        edge_type = yuid_mailru_barnavig
                    else:
                        raise Exception("Not implemented yet")
                    yield make_soup_rec(yuid, id_value, edge_type, dt)
            except ValueError:
                return

        decoded_ckpa = get_field_value("decoded_ckpa", http_params, "&")
        if decoded_ckpa:
            try:
                decoded_ckpa = urllib.unquote(decoded_ckpa)
                for neighbour_browser in json.loads(decoded_ckpa):
                    neighbour_yandexuid = neighbour_browser.get("yandexuid")
                    if neighbour_yandexuid is not None and yuid != neighbour_yandexuid:
                        yield make_soup_rec(yuid, neighbour_yandexuid, yuid_yuid_yabro_cookie_mining, dt)
            except ValueError:
                return

        if r1:
            yield make_soup_rec(yuid, r1, yuid_r1_barnavig, dt)

        if orig_ui and uuid:
            # TODO: put into error table: UI is a desktop identifier and UUID is mobile
            # yield rec
            pass
        elif orig_ui:  # desktop browser
            ui = parse_uuid(orig_ui)
            if ui:
                yield make_soup_rec(yuid, ui, yuid_ui_barnavig, dt)

            else:  # to track parsing errors
                # TODO: put into error table: bad UI
                # yield {'orig_ui': orig_ui, 'yuid': yuid, '@table_index': 5}
                pass
        elif uuid:
            yield make_soup_rec(yuid, uuid, yuid_uuid_barnavig, dt)


def parse_uuid(ui):
    if len(ui) < 32:
        return ""

    if "%12" in ui:  # some frequent specific case
        return ""

    if ui[0:3] == "%7B":  # has encoded brace
        ui = ui[3:-3]

    if ui[0:1] == "{":
        ui = ui[1:-1]

    if len(ui) == 32:  # format without dashes
        ui = ui[0:8] + "-" + ui[8:12] + "-" + ui[12:16] + "-" + ui[16:20] + "-" + ui[20:32]

    if len(ui) == 36:
        return ui.upper()
    else:
        return ""


def parse_mailru_social_ajax(ajax, resource, network="vk.com", id_prefix="vk"):
    id_name = id_prefix + "_id"
    netloc = urlparse(resource).netloc
    if netloc == network:
        ajax_query = urlparse(ajax)
        id_param = ([qparam for qparam in ajax_query.query.split("&") if qparam.startswith(id_name + "=")] or [None])[
            0
        ]
        if ajax_query.netloc in ["ad.mail.ru", "r3.mail.ru", "vk-callback.go.mail.ru"] and id_param:
            id_value = id_param.split("=")[1]
            return [id_value]
    return []


def parse_mailru_email_ajax(ajax, resource):
    source = urlparse(resource).netloc
    if source.endswith(".mail.ru") or source == "mail.ru":
        ajax_query = urlparse(ajax)
        user_param = (
            [qparam for qparam in urllib.unquote(ajax_query.query).split("&") if qparam.startswith("user=")] or [None]
        )[0]
        if ajax_query.netloc == "filin.mail.ru" and user_param:
            id_value = user_param.split("=")[1]
            return [id_value]
    return []


def parse_ajax(decoded_ajax, parsers):
    ajax_series = urllib.unquote(decoded_ajax)
    for site_ajax in ajax_series.strip("[]").split("],["):  # WARNING [[]]
        netloc = urlparse(site_ajax.split(",")[0].strip('"')).netloc
        for id_type, (social_resource, parser) in parsers.iteritems():
            if netloc.endswith(social_resource):
                resource = site_ajax.split(",")[0].strip('"')
                for ajax_request in site_ajax.split(",")[1:]:
                    for id_value in parser(ajax_request.strip('"'), resource):
                        if bool(id_value):
                            yield (id_value, id_type)


def reduce_soup(key, recs):
    dts = set()
    res = dict(**key)
    for r in recs:
        dts.add(r["dates"][0])
    res["dates"] = list(dts)
    yield res


def run_bar_navig(yt_client, input_tables, output_table):
    mapper = partial(
        map_bar_log,
        yuid_vkcom_barnavig=edges.get_edge_type(
            ids.YANDEXUID, ids.VK_ID, source_type.YABRO_AJAX, log_source.BAR_NAVIG_LOG
        ),
        yuid_okru_barnavig=edges.get_edge_type(
            ids.YANDEXUID, ids.OK_ID, source_type.YABRO_AJAX, log_source.BAR_NAVIG_LOG
        ),
        yuid_mailru_barnavig=edges.get_edge_type(
            ids.YANDEXUID, ids.EMAIL, source_type.YABRO_AJAX, log_source.BAR_NAVIG_LOG
        ),
        yuid_r1_barnavig=edges.get_edge_type(
            ids.YANDEXUID, ids.DISTR_R1, source_type.YASOFT, log_source.BAR_NAVIG_LOG
        ),
        yuid_ui_barnavig=edges.get_edge_type(
            ids.YANDEXUID, ids.DISTR_UI, source_type.YASOFT, log_source.BAR_NAVIG_LOG
        ),
        yuid_uuid_barnavig=edges.get_edge_type(
            ids.YANDEXUID, ids.UUID, source_type.YABRO_ANDROID, log_source.BAR_NAVIG_LOG
        ),
        yuid_yuid_yabro_cookie_mining=edges.get_edge_type(
            ids.YANDEXUID, ids.YANDEXUID, source_type.YABRO_COOKIE_MINING, log_source.BAR_NAVIG_LOG
        ),
        ajax_parsers={
            VK: ("vk.com", partial(parse_mailru_social_ajax, network="vk.com", id_prefix="vk")),
            OK: ("ok.ru", partial(parse_mailru_social_ajax, network="ok.ru", id_prefix="ok")),
            EMAIL: ("mail.ru", parse_mailru_email_ajax),
        },
    )

    reduce_by = ["id1", "id1Type", "id2", "id2Type", "sourceType", "logSource"]
    yt_client.run_map_reduce(mapper, reduce_soup, input_tables, output_table, reduce_by=reduce_by)


class BarNavigImportTask(StreamBaseYqlTask):

    log_source = log_source.BAR_NAVIG_LOG

    @property
    def query_template(self):
        return "bar.sql.j2"

    def observed_paths(self):
        yield conf.paths.logfeller.bar_navig_log.stream

    def run(self, *args, **kwargs):
        with self.yt.TempTable() as tmp:
            run_bar_navig(self.yt, self.unprocessed_tables, tmp)
            kwargs.update(tmp_source_path=str(tmp))
            super(BarNavigImportTask, self).run(*args, **kwargs)
