from functools import partial

from qb2.api.v1.typing import String, List
from nile.api.v1 import extractors as ne, clusters, Record, with_hints

from crypta.graph.soup.config.python import (  # noqa
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)
from crypta.graph.data_import.stream.lib.tasks.base import StreamBaseTask
import crypta.lib.python.bt.conf.conf as conf


PUNTO_BROWSERS = ["yu", "yu_ch", "yu_opch", "yu_yb", "yu_amg", "yu_edge", "yu_ff", "yu_op", "yu_op64", "yu_ie"]


def extract(field):
    def func(params):
        if params:
            res = params.get(field)
            if isinstance(res, list):
                return res[0]
            else:
                return res

    return func


def map_eal(recs, out_soup, type_eal, type_punto, type_extbro, need_dates=True):
    def mk_soup(yuid, ui, typ, dt):
        d = {
            "id1Type": typ.Id1Type.Name,
            "id1": yuid,
            "id2Type": typ.Id2Type.Name,
            "id2": ui,
            "sourceType": typ.SourceType.Name,
            "logSource": typ.LogSource.Name,
        }
        if need_dates:
            d["dates"] = [dt]
        return Record.from_dict(d)

    for r in recs:
        if r.ui:
            ui = r.ui.upper()  # qb2 gives lowercased uis, and we've been using uppercase ones so far
            if r.req_yuid:
                out_soup(mk_soup(r.req_yuid, ui, type_eal, r.date))

            if r.cookie_yuid and r.cookie_yuid != r.req_yuid:
                out_soup(mk_soup(r.cookie_yuid, ui, type_eal, r.date))

            if r.distr_yuid and r.distr_yuid != r.cookie_yuid and r.distr_yuid != r.req_yuid:
                out_soup(mk_soup(r.distr_yuid, ui, type_eal, r.date))

            if r.yasoft in ("punto", "puntomac"):
                for yu_source in PUNTO_BROWSERS:
                    yuid = getattr(r, yu_source)
                    if yuid:
                        out_soup(mk_soup(yuid, ui, type_punto, r.date))

            if r.ext_bro_cookies:
                yuids = [y for y in r.ext_bro_cookies.split(".") if y]
                if yuids:
                    for y in yuids:
                        out_soup(mk_soup(y, ui, type_extbro, r.date))


def local_run(input_file, out_mapped_qb2, out_extracted_qb2, out_soup):
    from nile.api.v1.local import FileSource, ListSink

    job = construct_nile_job("log_table", "out_soup", clusters.Hahn())
    job.local_run(
        sources={"log_tables": FileSource(input_file, format="json")},
        sinks={
            "mapped_qb2": ListSink(out_mapped_qb2),
            "extracted_qb2": ListSink(out_extracted_qb2),
            "out_soup": ListSink(out_soup),
        },
    )
    return job


def construct_nile_job(log_tables, out_soup_tbl, cluster, need_dates=True):
    job = cluster.job("Parse export-access-log day")

    extractors = {ys: ne.custom(extract(ys), "parsed_parameters").add_hints(type=str) for ys in PUNTO_BROWSERS}
    extractors.update(
        dict(
            req_yuid=ne.custom(extract("yandexuid"), "parsed_parameters").add_hints(type=str),
            cookie_yuid=ne.custom(extract("yandexuid"), "parsed_cookies").add_hints(type=str),
            distr_yuid=ne.custom(extract("distr_yandexuid"), "parsed_parameters").add_hints(type=str),
            ext_bro_cookies=ne.custom(extract("ckp"), "parsed_parameters").add_hints(type=str),
        )
    )

    stuff = (
        job.concat(*[job.table(x) for x in log_tables])
        .label("log_tables")
        .qb2(log="export-access-log", fields=["parsed_parameters", "parsed_cookies", "yasoft", "ui", "date"])
        .label("mapped_qb2")
        .project("yasoft", "ui", "date", **extractors)
        .label("extracted_qb2")
    )

    mapper = partial(
        map_eal,
        type_eal=edges.get_edge_type(ids.YANDEXUID, ids.DISTR_UI, source_type.YASOFT, log_source.EXPORT_ACCESS_LOG),
        type_punto=edges.get_edge_type(
            ids.YANDEXUID, ids.DISTR_UI, source_type.YASOFT_PUNTO_SWITCHER, log_source.EXPORT_ACCESS_LOG
        ),
        type_extbro=edges.get_edge_type(
            ids.YANDEXUID, ids.DISTR_UI, source_type.YABRO_EXTERNAL_BROWSERS, log_source.EXPORT_ACCESS_LOG
        ),
        need_dates=need_dates,
    )

    soup_schema = {
        "id1": String,
        "id1Type": String,
        "id2": String,
        "id2Type": String,
        "sourceType": String,
        "logSource": String,
    }

    if need_dates:
        soup_schema["dates"] = List[String]

    mapper = with_hints(outputs_count=1, output_schema=soup_schema)(mapper)

    out_soup = stuff.map(mapper, intensity="data")

    out_soup.label("out_soup").put(out_soup_tbl)

    return job


def run_eal(yt_proxy, yt_token, yt_pool, input_tables, output_table, need_dates=True):
    cluster = clusters.YT(proxy=yt_proxy, token=yt_token, pool=yt_pool)

    job = construct_nile_job(input_tables, output_table, cluster, need_dates)
    job.run()


class EalImportTask(StreamBaseTask):

    # nile got lock conflict if all task wrapped as transaction
    # so skip and do not wrap all within transaction
    NO_USE_TRANSACTION = True

    log_source = log_source.EXPORT_ACCESS_LOG

    def observed_paths(self):
        yield conf.paths.logfeller.export_access_log.stream

    def run(self, *args, **kwargs):
        if not self._assert_tables_ready(self.unprocessed_tables, self.input_tables):
            return

        run_eal(
            self.yt_proxy,
            self.get_token(),
            self.pool or conf.yt.pool or "research",
            self.unprocessed_tables,
            self.output_table,
        )
