#include "algo.h"

namespace NCrypta::NGraphEngine::NAlgo {
    using NCrypta::NGraphEngine::TGraphStats;

    void MarkDepthsAndBridges(ui64 vertex, ui64 prevVertex, TVector<i64>& depth, TVector<i64>& upDepth,
                     TVector<NInnerView::TEdge>& bridges, const NInnerView::TGraph& graph) {
        depth[vertex] = (prevVertex == vertex ? 0 : depth[prevVertex] + 1);
        upDepth[vertex] = depth[vertex];

        for (const auto& edge : graph.Edges[vertex]) {
            ui64 to = edge.Vertex2;
            if (to == prevVertex) {
                continue;
            }
            if (depth[to] < 0) {
                MarkDepthsAndBridges(to, vertex, depth, upDepth, bridges, graph);
                upDepth[vertex] = Min(upDepth[vertex], upDepth[to]);
                if (upDepth[to] > depth[vertex]) {
                    bridges.push_back({to, vertex});
                }
            } else {
                upDepth[vertex] = Min(upDepth[vertex], depth[to]);
            }
        }
    }

    TVector<NInnerView::TEdge> FindBridges(const NInnerView::TGraph& graph) {
        const auto size = graph.Vertices.size();
        TVector<i64> depth(size, -1);
        TVector<i64> upDepth(size, 0);
        TVector<NInnerView::TEdge> bridges;
        for (ui64 vertex = 0; vertex < size; ++vertex) {
            if (depth[vertex] < 0) {
                MarkDepthsAndBridges(vertex, vertex, depth, upDepth, bridges, graph);
            }
        }
        return bridges;
    }

    ui64 MarkComponents(ui64 vertex, ui64 label, ui64& maxLabel, TVector<ui64>& components, const TVector<i64>& depth, const TVector<i64>& upDepth, const NInnerView::TGraph& graph) {
        components[vertex] = label;
        for (const auto& edge : graph.Edges[vertex]) {
            ui64 to = edge.Vertex2;
            if (components[to]) {
                continue;
            }
            if (upDepth[to] > depth[vertex]) {
                maxLabel++;
                MarkComponents(to, maxLabel, maxLabel, components, depth, upDepth, graph);
            } else {
                MarkComponents(to, label, maxLabel, components, depth, upDepth, graph);
            }
        }
        return label;
    }

    TVector<ui64> FindTwoEdgeConnectedComponents(const NInnerView::TGraph& graph) {
        const auto size = graph.Vertices.size();
        TVector<i64> depth(size, -1);
        TVector<i64> upDepth(size, 0);
        TVector<NInnerView::TEdge> bridges;
        for (ui64 vertex = 0; vertex < size; ++vertex) {
            if (depth[vertex] < 0) {
                MarkDepthsAndBridges(vertex, vertex, depth, upDepth, bridges, graph);
            }
        }

        TVector<ui64> components(size, 0);
        ui64 maxLabel = 0;
        for (ui64 vertex = 0; vertex < size; ++vertex) {
            if (!components[vertex]) {
                maxLabel++;
                MarkComponents(vertex, maxLabel, maxLabel, components, depth, upDepth, graph);
            }
        }

        for (ui64 i = 0; i < size; ++i) {
            components[i]--;
        }

        return components;
    }

    TVector<NCustomView::TEdge> FindBridges(const NCrypta::NGraphEngine::TCommonGraph& graph) {
        return graph.ConvertToCustomView(FindBridges(graph.GetInnerGraph()));
    }

    TClusteringCoefficient ComputeClusteringCoefficient(const TCommonGraph& graph, const THumanMatchingTypes& types) {
        ui64 triangles = 0;
        ui64 allPossibleTriangles = 0;

        for (const auto &edges : graph.GetInnerGraph().Edges) {
            for (const auto &first : edges) {
                ui64 vertex = first.Vertex1;
                ui64 neighbour1 = first.Vertex2;
                for (const auto &second : edges) {
                    ui64 neighbour2 = second.Vertex2;
                    if (neighbour1 >= neighbour2) {
                        continue;
                    }
                    if (!types.IsAvailable(graph.ConvertToCustomView(neighbour1).Type, graph.ConvertToCustomView(neighbour2).Type)) {
                        continue;
                    }
                    if (graph.GetInnerGraph().IsAdjacent(neighbour1, neighbour2)) {
                        if (vertex < neighbour1) {
                            ++triangles;
                            ++allPossibleTriangles;
                        }
                    } else {
                        ++allPossibleTriangles;
                    }
                }
            }
        }

        TClusteringCoefficient coef;
        coef.SetTrianglesCount(triangles);
        coef.SetConnectedTripletsCount(allPossibleTriangles);
        double clusteringCoefficient = 0.;
        if (allPossibleTriangles > 0.) {
            clusteringCoefficient = (triangles + 0.) / (allPossibleTriangles + 0.);
        }
        coef.SetClusteringCoefficient(clusteringCoefficient);
        return coef;
    }

    TClusteringCoefficient ComputeClusteringCoefficient(const TCommonGraph& graph) {
        THumanMatchingTypes types{};
        return ComputeClusteringCoefficient(graph, types);
    }

    TGraphStats ComputeGraphStats(const TCommonGraph& graph) {
        auto cc = NCrypta::NGraphEngine::NAlgo::ComputeClusteringCoefficient(graph);
        TGraphStats stats;
        ui64 edgesCount = 0;
        for (const auto& edges : graph.GetInnerGraph().Edges) {
            edgesCount += edges.size();
        }
        edgesCount /= 2;
        stats.SetVerticesCount(graph.size());
        stats.SetEdgesCount(edgesCount);
        stats.SetBridgesCount(FindBridges(graph).size());
        stats.MutableCC()->MergeFrom(cc);
        return stats;
    }
}
