#include "graph.h"

namespace NCrypta::NGraphEngine {
    TCommonGraph::TCommonGraph(const TGraph& graph, bool forceEdgesStrong) {
        for (const auto& protoVertex : graph.GetVertices()) {
            NIdentifiers::TGenericID genericId(protoVertex);
            NCustomView::TVertex vertex{genericId.GetType(), genericId.GetValue()};
            AddVertex(vertex);
        }
        for (const auto& edge : graph.GetEdges()) {
            AddEdge(edge.GetVertex1(), edge.GetVertex2(), edge, forceEdgesStrong);
        }
    }

    void TCommonGraph::AddEdge(ui64 firstVertexId, ui64 secondVertexId, const NCrypta::NGraphEngine::TEdge& edge, bool forceEdgeStrong) {
        NInnerView::TEdge nativeEdge = {
                .Vertex1=firstVertexId,
                .Vertex2=secondVertexId,
                .Weight = edge.GetSurvivalWeight(),
                .IsStrong = (forceEdgeStrong ? true : edge.GetIsStrong())
        };
        Graph.Edges[firstVertexId].push_back(nativeEdge);
        nativeEdge.Vertex1 = secondVertexId;
        nativeEdge.Vertex2 = firstVertexId;
        Graph.Edges[secondVertexId].push_back(nativeEdge);
    }

    void TCommonGraph::AddEdges(const TVector<NCustomView::TEdge>& edges) {
        for (const auto &edge : edges) {
            AddEdge(edge);
        }
    }

    void TCommonGraph::AddEdge(const NCustomView::TEdge& edge) {
        AddEdge(GetOrCreateIndex(edge.Vertex1), GetOrCreateIndex(edge.Vertex2), edge.Edge);
    }

    ui64 TCommonGraph::AddVertex(const NCustomView::TVertex& vertex) {
        return GetOrCreateIndex(vertex);
    }

    ui64 TCommonGraph::size() const {
        return Graph.Edges.ysize();
    }

    const NInnerView::TGraph& TCommonGraph::GetInnerGraph() const {
        return Graph;
    }

    bool NInnerView::TGraph::IsAdjacent(ui64 vertex1, ui64 vertex2) const {
        for (auto edge : Edges[vertex1]) {
            if (edge.Vertex2 == vertex2) {
                return true;
            }
        }
        return false;
    }

    TVector<NCustomView::TEdge> TCommonGraph::ConvertToCustomView(const TVector<NInnerView::TEdge>& edges) const {
        TVector<NCrypta::NGraphEngine::NCustomView::TEdge> result;
        result.reserve(edges.size());
        for (const auto &edge : edges) {
            TEdge protoEdge;
            protoEdge.SetIsStrong(edge.IsStrong);
            protoEdge.SetSurvivalWeight(edge.Weight);
            NCustomView::TEdge customEdge{
                    Graph.Vertices.at(edge.Vertex1).CustomVertex,
                    Graph.Vertices.at(edge.Vertex2).CustomVertex,
                    protoEdge};
            result.emplace_back(customEdge);
        }
        return result;
    }

    NCustomView::TVertex TCommonGraph::ConvertToCustomView(ui64 vertex) const {
        return Graph.Vertices.at(vertex).CustomVertex;
    }

    TVector<NCustomView::TVertex> TCommonGraph::ConvertToCustomView(const TVector<NInnerView::TVertex>& vertices) const {
        TVector<NCustomView::TVertex> result;
        result.reserve(vertices.size());
        for (const auto &vertex : vertices) {
            result.push_back(ConvertToCustomView(vertex.Index));
        }
        return result;
    }


    ui64 TCommonGraph::GetOrCreateIndex(const NCustomView::TVertex& vertex) {
        if (const auto vertexIt = Graph.IdToVertexIndex.find(vertex); vertexIt != Graph.IdToVertexIndex.end()) {
            return vertexIt->second;
        }
        ui64 index = static_cast<ui64>(Graph.Edges.size());
        Graph.Edges.push_back({});
        Graph.IdToVertexIndex[vertex] = index;
        Graph.Vertices[index] = {index, vertex};
        return index;
    }
}

template<>
void Out<NCrypta::NGraphEngine::NInnerView::TGraph>(IOutputStream& out, const NCrypta::NGraphEngine::NInnerView::TGraph& graph) {
    out << "[";
    for (const auto& vertex : graph.Vertices) {
        out << vertex.second << "  ";
    }
    out << "]" << Endl;
    ui64 edgesCount = 0;
    for (const auto& edges: graph.Edges) {
        for (const auto& edge : edges) {
            out << edge << " ";
            edgesCount++;
        }
        out << Endl;
    }
    out << "vertices: " << graph.Vertices.size()
        << ", "
        << "edges: "  << edgesCount / 2 << Endl;

}

template<>
void Out<NCrypta::NGraphEngine::NInnerView::TVertex>(IOutputStream& out, const NCrypta::NGraphEngine::NInnerView::TVertex& vertex) {
    out << vertex.Index << ": " << vertex.CustomVertex;
}

template<>
void Out<NCrypta::NGraphEngine::NCustomView::TVertex>(IOutputStream& out, const NCrypta::NGraphEngine::NCustomView::TVertex& vertex) {
    out << "(" << NCrypta::NSoup::IdType(vertex.Type).GetName() << " " << vertex.Value << ")";
}

template<>
void Out<NCrypta::NGraphEngine::NInnerView::TEdge>(IOutputStream& out, const NCrypta::NGraphEngine::NInnerView::TEdge& edge) {
    out << "(" << edge.Vertex1 << " " << edge.Vertex2 << ")";
}
