#include "hm.h"
#include <util/random/random.h>

namespace NCrypta::NGraphEngine {
    THumanMatchingTypes::THumanMatchingTypes() {
        Init();
    }

    bool THumanMatchingTypes::IsAvailable(const NCrypta::NSoup::TEdgeType &type) const {
        return IsAvailable(type.GetId1Type(), type.GetId2Type());
    }

    bool THumanMatchingTypes::IsAvailable(const EIdType &type1, const EIdType &type2) const {
        auto typesIt = Neighbours.find(type1);
        if (typesIt != Neighbours.end()) {
            return typesIt->second.find(type2) != typesIt->second.end();
        }
        return false;
    }

    const TSet<EIdType>& THumanMatchingTypes::GetAdjacentTypes(const EIdType &type) const {
        if (const auto it{Neighbours.find(type)}; it != Neighbours.end()) {
            return it->second;
        }
        return Empty;
    }

    void THumanMatchingTypes::Init() {
        for (const auto &record : NCrypta::NSoup::EdgeRecords()) {
            if (record.GetUsage().GetHumanMatching()) {
                Neighbours[record.GetType().GetId1Type()].insert(record.GetType().GetId2Type());
                Neighbours[record.GetType().GetId2Type()].insert(record.GetType().GetId1Type());
            }
        }
    }

    NCrypta::NGraphEngine::TCommonGraph THMGraphGenerator::GenerateGraph(ui64 size, double connectivityRate) {
        NCrypta::NGraphEngine::TCommonGraph graph;
        TMap<EIdType, TVector<ui64>> vertices;
        NCrypta::NGraphEngine::THumanMatchingTypes types{};

        {
            auto vertex = GenerateVertex(EIdType::YANDEXUID);
            vertices[vertex.Type].push_back(graph.size());
            graph.AddVertex(vertex);
        }
        for (ui64 i = 0; i < size; ++i) {
            const auto &vertex = ChooseRandomVertex(graph);
            auto& adjacentTypes = types.GetAdjacentTypes(vertex.Type);
            if (!adjacentTypes.size()) {
                continue;
            }
            auto typeIt = adjacentTypes.begin();
            std::advance(typeIt, RandomNumber<ui64>(adjacentTypes.size()));
            const auto& indexies = vertices.find(*typeIt);
            if ((RandomNumber<double>() < connectivityRate) && (indexies != vertices.end())) {
                auto indexIt = indexies->second.begin();
                std::advance(indexIt, RandomNumber<ui64>(indexies->second.size()));
                graph.AddEdge({vertex, graph.ConvertToCustomView(*indexIt)});

            } else {
                vertices[*typeIt].push_back(graph.size());
                graph.AddEdge({vertex, GenerateVertex(*typeIt)});
            }
        }
        return graph;
    }

    NCrypta::NGraphEngine::NCustomView::TVertex THMGraphGenerator::ChooseRandomVertex(const NCrypta::NGraphEngine::TCommonGraph& graph) {
        return graph.ConvertToCustomView(RandomNumber<ui64>(graph.size()));
    }

    NCrypta::NGraphEngine::NCustomView::TVertex THMGraphGenerator::GenerateVertex(const EIdType& type) {
        return {
                type,
                NIdentifiers::TGenericID::Next(type)
        };
    }
}
