#include <library/cpp/testing/unittest/registar.h>
#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/graph/engine/graph/lib/graph.h>
#include <crypta/graph/engine/graph/lib/split.h>



Y_UNIT_TEST_SUITE(TUnitSplitTest) {

    using NCrypta::NGraphEngine::TCommonGraph;

    Y_UNIT_TEST(TSplitSimpleTest) {
        TCommonGraph graph;
        graph.AddEdges({
                {
                        {EIdType::YANDEXUID, "1"},
                        {EIdType::YANDEXUID, "2"}
                },
                {
                        {EIdType::YANDEXUID, "2"},
                        {EIdType::YANDEXUID, "3"}
                },
                {
                        {EIdType::YANDEXUID, "3"},
                        {EIdType::YANDEXUID, "1"}
                },
                {
                        {EIdType::YANDEXUID, "3"},
                        {EIdType::YANDEXUID, "4"}
                },
                {
                        {EIdType::YANDEXUID, "4"},
                        {EIdType::YANDEXUID, "5"}
                },
                {
                        {EIdType::YANDEXUID, "5"},
                        {EIdType::YANDEXUID, "6"}
                },
                {
                        {EIdType::YANDEXUID, "6"},
                        {EIdType::YANDEXUID, "4"}
                },
                {
                        {EIdType::YANDEXUID, "6"},
                        {EIdType::YANDEXUID, "7"}
                }
        });


        const auto split = NCrypta::NGraphEngine::Split(graph);

        UNIT_ASSERT_EQUAL(split.Split.size(),  2);
        UNIT_ASSERT_EQUAL(split.Split.front().size(),  3); // 1 2 3
        UNIT_ASSERT_EQUAL(split.Split.back().size(),  4);  // 4 5 6 7
    }
    Y_UNIT_TEST(TSplitLeafVertexeTest) {
        TCommonGraph graph;

        NCrypta::NGraphEngine::TEdge protoEdge1;
        protoEdge1.SetSurvivalWeight(0.0000001);
        protoEdge1.SetIsStrong(false);
        NCrypta::NGraphEngine::NCustomView::TEdge E1;
        E1.Vertex1 = {EIdType::YANDEXUID, "1"};
        E1.Vertex2 = {EIdType::YANDEXUID, "2"};
        E1.Edge = protoEdge1;

        NCrypta::NGraphEngine::TEdge protoEdge2;
        protoEdge2.SetSurvivalWeight(1);
        protoEdge2.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E2;
        E2.Vertex1 = {EIdType::YANDEXUID, "2"};
        E2.Vertex2 = {EIdType::YANDEXUID, "3"};
        E2.Edge = protoEdge2;

        NCrypta::NGraphEngine::TEdge protoEdge3;
        protoEdge3.SetSurvivalWeight(1);
        protoEdge3.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E3;
        E3.Vertex1 = {EIdType::YANDEXUID, "3"};
        E3.Vertex2 = {EIdType::YANDEXUID, "4"};
        E3.Edge = protoEdge3;

        NCrypta::NGraphEngine::TEdge protoEdge4;
        protoEdge4.SetSurvivalWeight(1);
        protoEdge4.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E4;
        E4.Vertex1 = {EIdType::YANDEXUID, "2"};
        E4.Vertex2 = {EIdType::YANDEXUID, "4"};
        E4.Edge = protoEdge4;

        graph.AddEdges({E1, E2, E3, E4});

        const auto split = NCrypta::NGraphEngine::Split(graph);

        UNIT_ASSERT_EQUAL(split.Split.size(), 1);
        UNIT_ASSERT_EQUAL(split.Split.front().size(), 4); // all
    }

    Y_UNIT_TEST(TSplitBadVertexBetweenGoodComponents) {
        TCommonGraph graph;

        NCrypta::NGraphEngine::TEdge protoEdge1;
        protoEdge1.SetSurvivalWeight(0.0000001);
        protoEdge1.SetIsStrong(false);
        NCrypta::NGraphEngine::NCustomView::TEdge E1;
        E1.Vertex1 = {EIdType::YANDEXUID, "1"};
        E1.Vertex2 = {EIdType::YANDEXUID, "2"};
        E1.Edge = protoEdge1;

        NCrypta::NGraphEngine::TEdge protoEdge2;
        protoEdge2.SetSurvivalWeight(1);
        protoEdge2.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E2;
        E2.Vertex1 = {EIdType::YANDEXUID, "2"};
        E2.Vertex2 = {EIdType::YANDEXUID, "3"};
        E2.Edge = protoEdge2;

        NCrypta::NGraphEngine::TEdge protoEdge3;
        protoEdge3.SetSurvivalWeight(1);
        protoEdge3.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E3;
        E3.Vertex1 = {EIdType::YANDEXUID, "3"};
        E3.Vertex2 = {EIdType::YANDEXUID, "4"};
        E3.Edge = protoEdge3;

        NCrypta::NGraphEngine::TEdge protoEdge4;
        protoEdge4.SetSurvivalWeight(1);
        protoEdge4.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E4;
        E4.Vertex1 = {EIdType::YANDEXUID, "2"};
        E4.Vertex2 = {EIdType::YANDEXUID, "4"};
        E4.Edge = protoEdge4;

        NCrypta::NGraphEngine::TEdge protoEdge5;
        protoEdge5.SetSurvivalWeight(0.0000001);
        protoEdge5.SetIsStrong(false);
        NCrypta::NGraphEngine::NCustomView::TEdge E5;
        E5.Vertex1 = {EIdType::YANDEXUID, "1"};
        E5.Vertex2 = {EIdType::YANDEXUID, "5"};
        E5.Edge = protoEdge5;

        NCrypta::NGraphEngine::TEdge protoEdge6;
        protoEdge6.SetSurvivalWeight(1);
        protoEdge6.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E6;
        E6.Vertex1 = {EIdType::YANDEXUID, "5"};
        E6.Vertex2 = {EIdType::YANDEXUID, "6"};
        E6.Edge = protoEdge6;

        NCrypta::NGraphEngine::TEdge protoEdge7;
        protoEdge7.SetSurvivalWeight(1);
        protoEdge7.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E7;
        E7.Vertex1 = {EIdType::YANDEXUID, "6"};
        E7.Vertex2 = {EIdType::YANDEXUID, "7"};
        E7.Edge = protoEdge7;

        NCrypta::NGraphEngine::TEdge protoEdge8;
        protoEdge8.SetSurvivalWeight(1);
        protoEdge8.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E8;
        E8.Vertex1 = {EIdType::YANDEXUID, "5"};
        E8.Vertex2 = {EIdType::YANDEXUID, "7"};
        E8.Edge = protoEdge8;

        graph.AddEdges({E1, E2, E3, E4, E5, E6, E7, E8});

        const auto split = NCrypta::NGraphEngine::Split(graph);

        UNIT_ASSERT_EQUAL(split.Split.size(), 2);
    }

    Y_UNIT_TEST(TJoinSmallComponentsBetweenBigComponentsTest) {
        NCrypta::NGraphEngine::TCommonGraph graph{};

        NCrypta::NGraphEngine::TEdge protoEdge1;
        protoEdge1.SetSurvivalWeight(1);
        protoEdge1.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E1;
        E1.Vertex1 = {EIdType::YANDEXUID, "0"};
        E1.Vertex2 = {EIdType::YANDEXUID, "1"};
        E1.Edge = protoEdge1;

        NCrypta::NGraphEngine::TEdge protoEdge2;
        protoEdge2.SetSurvivalWeight(1);
        protoEdge2.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E2;
        E2.Vertex1 = {EIdType::YANDEXUID, "0"};
        E2.Vertex2 = {EIdType::YANDEXUID, "2"};
        E2.Edge = protoEdge2;

        NCrypta::NGraphEngine::TEdge protoEdge3;
        protoEdge3.SetSurvivalWeight(1);
        protoEdge3.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E3;
        E3.Vertex1 = {EIdType::YANDEXUID, "1"};
        E3.Vertex2 = {EIdType::YANDEXUID, "2"};
        E3.Edge = protoEdge3;

        NCrypta::NGraphEngine::TEdge protoEdge4;
        protoEdge4.SetSurvivalWeight(0.001);
        protoEdge4.SetIsStrong(false);
        NCrypta::NGraphEngine::NCustomView::TEdge E4;
        E4.Vertex1 = {EIdType::YANDEXUID, "0"};
        E4.Vertex2 = {EIdType::YANDEXUID, "3"};
        E4.Edge = protoEdge4;

        NCrypta::NGraphEngine::TEdge protoEdge5;
        protoEdge5.SetSurvivalWeight(0.8);
        protoEdge5.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E5;
        E5.Vertex1 = {EIdType::YANDEXUID, "3"};
        E5.Vertex2 = {EIdType::YANDEXUID, "4"};
        E5.Edge = protoEdge5;

        NCrypta::NGraphEngine::TEdge protoEdge6;
        protoEdge6.SetSurvivalWeight(0.001);
        protoEdge6.SetIsStrong(false);
        NCrypta::NGraphEngine::NCustomView::TEdge E6;
        E6.Vertex1 = {EIdType::YANDEXUID, "4"};
        E6.Vertex2 = {EIdType::YANDEXUID, "5"};
        E6.Edge = protoEdge6;

        NCrypta::NGraphEngine::TEdge protoEdge7;
        protoEdge7.SetSurvivalWeight(1);
        protoEdge7.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E7;
        E7.Vertex1 = {EIdType::YANDEXUID, "5"};
        E7.Vertex2 = {EIdType::YANDEXUID, "6"};
        E7.Edge = protoEdge7;

        NCrypta::NGraphEngine::TEdge protoEdge8;
        protoEdge8.SetSurvivalWeight(1);
        protoEdge8.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E8;
        E8.Vertex1 = {EIdType::YANDEXUID, "6"};
        E8.Vertex2 = {EIdType::YANDEXUID, "7"};
        E8.Edge = protoEdge8;

        NCrypta::NGraphEngine::TEdge protoEdge9;
        protoEdge9.SetSurvivalWeight(1);
        protoEdge9.SetIsStrong(true);
        NCrypta::NGraphEngine::NCustomView::TEdge E9;
        E9.Vertex1 = {EIdType::YANDEXUID, "5"};
        E9.Vertex2 = {EIdType::YANDEXUID, "7"};
        E9.Edge = protoEdge9;


        graph.AddEdges({E1, E2, E3, E4, E5, E6, E7, E8, E9});
        auto innerGraph = graph.GetInnerGraph();

        TDisjointSets components(innerGraph.Vertices.size());

        //big component number 1
        components.UnionSets(0, 1);
        components.UnionSets(1, 2);

        //big component number 2
        components.UnionSets(5, 6);
        components.UnionSets(6, 7);

        components.UnionSets(3, 4); //not big enough

        NCrypta::NGraphEngine::JoinSmallComponentsBetweenBigComponents(innerGraph, components);

        auto split = NCrypta::NGraphEngine::ConvertComponentsToSplit(innerGraph.Vertices.size(), components);
        const auto splitSize = *MaxElement(split.begin(), split.end()) + 1;
        TVector<TVector<ui64>> customSplit(splitSize);

        for (ui64 vertex = 0; vertex < split.size(); ++vertex) {
            auto label = split[vertex];
            customSplit[label].push_back(vertex);
        }

        UNIT_ASSERT_EQUAL(splitSize, 2);
        UNIT_ASSERT_EQUAL(customSplit[0].size(), 5); // 0, 1, 2, 3, 4
        UNIT_ASSERT_EQUAL(customSplit[1].size(), 3);  // 5, 6, 7
    }
}
