#pragma once

#include <crypta/graph/engine/score/native/lib/base/strategy.h>
#include <crypta/graph/engine/score/native/lib/utils/getters.h>
#include <util/generic/map.h>
#include <util/generic/yexception.h>
#include <util/generic/hash_set.h>
#include <util/generic/ymath.h>

namespace NCrypta {
    namespace NGraphEngine {
        using namespace ::NCrypta::NIdentifiersProto::NIdType;

        // HistogramCountsScoringStrategy analog
        // https://a.yandex-team.ru/arc/trunk/arcadia/crypta/graph/matching/model/src/main/java/ru/yandex/crypta/graph2/model/matching/component/score/HistogramCountsScoringStrategy.java?rev=4954834#L62
        class THistogramCountsScore {
        private:
            struct TRange {
                ui64 From = 0;
                ui64 To = 0;
                double Value = 0.;
                double (*ProbFunc)(const TRange&, ui64) = UniformValue;

                static TRange BuildUniform(ui64 from, ui64 to, double value);
                static TRange BuildIncreasing(ui64 from, ui64 to, double value);
                static TRange BuildDecreasing(ui64 from, ui64 to, double value);

                double GetProbability(ui64 count) const;
                double GetProbabilityInterval() const;
            };

            static bool CheckInterval(const TRange& range, ui64 key);
            static double IncreasingValue(const TRange& range, ui64 count);
            static double DecreasingValue(const TRange& range, ui64 count);
            static double UniformValue(const TRange& range, ui64 count);

            struct TCounts {
                TMap<ui64, TRange> Ranges = {};

                double TheRestProb = 0.;
                bool PenalizeEmpty = false;
            };
            TCounts Counts{};

        public:
            THistogramCountsScore() {}
            THistogramCountsScore(const TCounts& counts) : Counts(counts) {}

            double GetScore(ui64 count) const;

            class Builder {
            public:
                Builder();

                void LessOrEqualAs(ui64 count, double prob);
                void UniformIncreasingRange(ui64 from, ui64 to, double probabilityRange);
                void UniformDecreasingRange(ui64 from, ui64 to, double probabilityRange);
                void AndTheRestAs(double restProb);
                void AndPenalizeEmpty();
                THistogramCountsScore Build();
            private:
                TCounts Counts{};
            };
        };

        class TAbstractCountScoringStrategy: public TAbstractScoringStrategy {
        public:
            struct TOptions {
                double ScoreWeight;
                THistogramCountsScore HistogramCountsScore{};
            };

            TAbstractCountScoringStrategy(const TString& name, const TOptions& options)
                : TAbstractScoringStrategy(name)
                , Options(options) {}
            TAbstractCountScoringStrategy() {}
            using TAbstractScoringStrategy::TAbstractScoringStrategy;
            virtual ~TAbstractCountScoringStrategy() = default;

            virtual ui64 Count(const TGraph& /*graph*/) const = 0;
            double ComputeScore(const TGraph& graph) const override;
            double ComputeWeight(const TGraph& /*graph*/) const override;
        protected:
            TOptions Options;
        };

        class TVerticesCountScoringStrategy: public TAbstractCountScoringStrategy {
        public:
            using TAbstractCountScoringStrategy::TAbstractCountScoringStrategy;
            ui64 Count(const TGraph& graph) const override;
        };

        class TDevicesCountStrategy: public TAbstractCountScoringStrategy {
        public:
            using TAbstractCountScoringStrategy::TAbstractCountScoringStrategy;
            ui64 Count(const TGraph& graph) const override;
        };

        class TTypesCountScoringStrategy: public TAbstractCountScoringStrategy {
        public:
            TTypesCountScoringStrategy(const TString& name, const EIdType& searchType, const TOptions& options)
                : TAbstractCountScoringStrategy(name, options)
                , SearchType(searchType) {}

            ui64 Count(const TGraph& graph) const override;
        private:
            EIdType SearchType{};
        };

        class TLoginsCountScoringStrategy : public TAbstractCountScoringStrategy {
        public:
            using TAbstractCountScoringStrategy::TAbstractCountScoringStrategy;
            ui64 Count(const TGraph& graph) const override;
        };

        class TCrossDevicesStrategy: public TAbstractScoringStrategy {
        public:

            using TAbstractScoringStrategy::TAbstractScoringStrategy;
            double ComputeScore(const TGraph& graph) const override;
        };

        class TSocdemScoringStrategy: public TAbstractScoringStrategy {
        public:
            using TAbstractScoringStrategy::TAbstractScoringStrategy;
            double ComputeScore(const TGraph& graph) const override;
        };
    }
}
