#pragma once

#include <crypta/graph/engine/proto/graph.pb.h>
#include <crypta/graph/engine/score/proto/stats.pb.h>
#include <crypta/graph/engine/score/native/lib/base/strategy.h>
#include "counts.h"
#include "lcs_scores.h"

namespace NCrypta {
    namespace NGraphEngine {
        TMultiScoringStrategy GetCountsStrategy(const TStatsOptions& options) {
            TMultiScoringStrategy scoresCollector{};
            {
                THistogramCountsScore::Builder devicesHist{};
                devicesHist.LessOrEqualAs(0, 0);
                devicesHist.LessOrEqualAs(1, 0.6);
                devicesHist.LessOrEqualAs(3, 0.3);
                devicesHist.UniformDecreasingRange(4, 20, 0.1);
                devicesHist.AndTheRestAs(0);
                TDevicesCountStrategy devicesScore(
                        "devices_count", {.ScoreWeight = 1., .HistogramCountsScore = devicesHist.Build()});
                scoresCollector.Add(MakeHolder<TDevicesCountStrategy>(devicesScore));
            }
            {
                TCrossDevicesStrategy strategy("cross_device", options.GetCrossDeviceWeight());
                scoresCollector.Add(MakeHolder<TCrossDevicesStrategy>(strategy));
            }
            {
                THistogramCountsScore::Builder loginsHist{};
                loginsHist.LessOrEqualAs(0, 0.);
                loginsHist.LessOrEqualAs(2, 0.6);
                loginsHist.LessOrEqualAs(3, 0.3);
                loginsHist.UniformDecreasingRange(4, 20, 0.1);
                loginsHist.AndTheRestAs(0.);

                TLoginsCountScoringStrategy loginScore(
                        "logins_count", {.ScoreWeight = 1., .HistogramCountsScore = loginsHist.Build()});

                scoresCollector.Add(MakeHolder<TLoginsCountScoringStrategy>(loginScore));
            }
            {
                THistogramCountsScore::Builder emailsHist{};
                emailsHist.LessOrEqualAs(0, 0.);
                emailsHist.LessOrEqualAs(3, 0.6);
                emailsHist.LessOrEqualAs(5, 0.3);
                emailsHist.UniformDecreasingRange(6, 20, 0.1);
                emailsHist.AndTheRestAs(0);

                TTypesCountScoringStrategy emailScore(
                        "emails_count", EIdType::EMAIL, {.ScoreWeight = 1., .HistogramCountsScore = emailsHist.Build()});
                scoresCollector.Add(MakeHolder<TTypesCountScoringStrategy>(emailScore));
            }
            {
                THistogramCountsScore::Builder phonesHist{};
                phonesHist.LessOrEqualAs(0, 0.);
                phonesHist.LessOrEqualAs(2, 0.6);
                phonesHist.LessOrEqualAs(3, 0.3);
                phonesHist.UniformDecreasingRange(4, 20, 0.1);
                phonesHist.AndTheRestAs(0.);

                TTypesCountScoringStrategy phoneScore(
                        "phones_count", EIdType::PHONE, {.ScoreWeight = 1., .HistogramCountsScore = phonesHist.Build()});
                scoresCollector.Add(MakeHolder<TTypesCountScoringStrategy>(phoneScore));
            }
            {
                THistogramCountsScore::Builder vkHist{};
                vkHist.LessOrEqualAs(0, 0.);
                vkHist.LessOrEqualAs(1, 0.6);
                vkHist.LessOrEqualAs(2, 0.3);
                vkHist.UniformDecreasingRange(3, 20, 0.1);
                vkHist.AndTheRestAs(0.);

                TTypesCountScoringStrategy phoneScore(
                        "vk_count", EIdType::VK_ID, {.ScoreWeight = 1., .HistogramCountsScore = vkHist.Build()});
                scoresCollector.Add(MakeHolder<TTypesCountScoringStrategy>(phoneScore));
            }

            {
                THistogramCountsScore::Builder okHist{};
                okHist.LessOrEqualAs(0, 0.);
                okHist.LessOrEqualAs(1, 0.6);
                okHist.LessOrEqualAs(2, 0.3);
                okHist.UniformDecreasingRange(3, 20, 0.1);
                okHist.AndTheRestAs(0.);

                TTypesCountScoringStrategy okScore(
                        "ok_count", EIdType::OK_ID, {.ScoreWeight = 1., .HistogramCountsScore = okHist.Build()});
                scoresCollector.Add(MakeHolder<TTypesCountScoringStrategy>(okScore));
            }
            {
                THistogramCountsScore::Builder verticesHist{};
                verticesHist.LessOrEqualAs(0, 0.);
                verticesHist.UniformIncreasingRange(0, 1000, 0.7);
                verticesHist.UniformDecreasingRange(1001, 10000, 0.3);
                verticesHist.AndTheRestAs(0);
                TVerticesCountScoringStrategy verticesCountScore(
                        "vertices_count", {.ScoreWeight = 100.,
                                .HistogramCountsScore = verticesHist.Build()});
                scoresCollector.Add(MakeHolder<TVerticesCountScoringStrategy>(verticesCountScore));
            }

            {
                TSocdemScoringStrategy strategy("socdem", options.GetSocdemWeight());
                scoresCollector.Add(MakeHolder<TSocdemScoringStrategy>(strategy));
            }

            return scoresCollector;
        }

        TStats CollectStats(const TGraph& graph, const TMultiScoringStrategy& scoringStrategy) {
            return scoringStrategy.ComputeGraphScores(graph).ToProto(graph.GetId());
        }

        // human matching stats
        TMultiScoringStrategy GetProdScoringStrategy(const TStatsOptions& options) {
            TMultiScoringStrategy scoresCollector = GetCountsStrategy(options);
            {
                THistogramCountsScore::Builder loginsLcsHist{};
                loginsLcsHist.LessOrEqualAs(1, 0.);
                loginsLcsHist.LessOrEqualAs(6, 0.12);
                loginsLcsHist.LessOrEqualAs(12, 0.19);
                loginsLcsHist.LessOrEqualAs(18, 0.29);
                loginsLcsHist.AndTheRestAs(0.4);
                TLoginsSimpleLcsScoringStrategy loginsLcsScore(
                        "logins_lcs", {.ScoreWeight = 1., .HistogramCountsScore = loginsLcsHist.Build()});
                scoresCollector.Add(MakeHolder<TLoginsSimpleLcsScoringStrategy>(loginsLcsScore));
            }
            {
                THistogramCountsScore::Builder emailsLcsHist{};
                emailsLcsHist.LessOrEqualAs(1, 0.);
                emailsLcsHist.LessOrEqualAs(7, 0.11);
                emailsLcsHist.LessOrEqualAs(14, 0.19);
                emailsLcsHist.LessOrEqualAs(21, 0.29);
                emailsLcsHist.AndTheRestAs(0.41);
                TEmailsSimpleLcsScoringStrategy emailsLCSScore(
                        "emails_lcs", {.ScoreWeight = 1., .HistogramCountsScore = emailsLcsHist.Build()});
                scoresCollector.Add(MakeHolder<TEmailsSimpleLcsScoringStrategy>(emailsLCSScore));
            }

            return scoresCollector;
        }

        TMultiScoringStrategy GetExpScoringStrategy(const TStatsOptions& options) {
            TMultiScoringStrategy scoresCollector = GetCountsStrategy(options);
            {
                THistogramCountsScore::Builder loginsLcsHist{};
                loginsLcsHist.LessOrEqualAs(1, 0.);
                loginsLcsHist.LessOrEqualAs(6, 0.12);
                loginsLcsHist.LessOrEqualAs(12, 0.19);
                loginsLcsHist.LessOrEqualAs(18, 0.29);
                loginsLcsHist.AndTheRestAs(0.4);
                TLoginsSimpleLcsScoringStrategy loginsLcsScore(
                        "logins_lcs", {.ScoreWeight = .2, .HistogramCountsScore = loginsLcsHist.Build()});
                scoresCollector.Add(MakeHolder<TLoginsSimpleLcsScoringStrategy>(loginsLcsScore));
            }
            {
                THistogramCountsScore::Builder emailsLcsHist{};
                emailsLcsHist.LessOrEqualAs(1, 0.);
                emailsLcsHist.LessOrEqualAs(7, 0.11);
                emailsLcsHist.LessOrEqualAs(14, 0.19);
                emailsLcsHist.LessOrEqualAs(21, 0.29);
                emailsLcsHist.AndTheRestAs(0.41);
                TEmailsSimpleLcsScoringStrategy emailsLCSScore(
                        "emails_lcs", {.ScoreWeight = .2, .HistogramCountsScore = emailsLcsHist.Build()});
                scoresCollector.Add(MakeHolder<TEmailsSimpleLcsScoringStrategy>(emailsLCSScore));
            }

            {
                THistogramCountsScore::Builder loginsLcsHist{};
                loginsLcsHist.LessOrEqualAs(1, 0.);
                loginsLcsHist.LessOrEqualAs(2, 0.12);
                loginsLcsHist.LessOrEqualAs(3, 0.15);
                loginsLcsHist.LessOrEqualAs(6, 0.2);
                loginsLcsHist.LessOrEqualAs(10, 0.3);
                loginsLcsHist.AndTheRestAs(0.4);
                TLoginsAverageBestLcsScoringStrategy loginsLcsScore(
                        "avg_logins_lcs", {.ScoreWeight = .8, .HistogramCountsScore = loginsLcsHist.Build()});
                scoresCollector.Add(MakeHolder<TLoginsAverageBestLcsScoringStrategy>(loginsLcsScore));
            }
            {
                THistogramCountsScore::Builder emailsLcsHist{};
                emailsLcsHist.LessOrEqualAs(1, 0.);
                emailsLcsHist.LessOrEqualAs(2, 0.11);
                emailsLcsHist.LessOrEqualAs(3, 0.15);
                emailsLcsHist.LessOrEqualAs(6, 0.2);
                emailsLcsHist.LessOrEqualAs(10, 0.3);
                emailsLcsHist.AndTheRestAs(0.4);
                TEmailsAverageBestLcsScoringStrategy emailsLCSScore(
                        "avg_emails_lcs", {.ScoreWeight = .8, .HistogramCountsScore = emailsLcsHist.Build()});
                scoresCollector.Add(MakeHolder<TEmailsAverageBestLcsScoringStrategy>(emailsLCSScore));
            }

            return scoresCollector;
        }

        TStats CollectProdStats(const TGraph& graph, const TStatsOptions& options) {
            TMultiScoringStrategy strategy = GetProdScoringStrategy(options);
            TScores scores = strategy.ComputeGraphScores(graph);
            return scores.ToProto(graph.GetId(), scores.GetWeightedScore());
        }

        TStats CollectExpStats(const TGraph& graph, const TStatsOptions& options) {
            TMultiScoringStrategy strategy = GetExpScoringStrategy(options);
            TScores scores = strategy.ComputeGraphScores(graph);
            return scores.ToProto(graph.GetId(), scores.GetWeightedScore());
        }
    }
}
