#pragma once

#include <google/protobuf/util/message_differencer.h>
#include <crypta/lib/python/native_yt/cpp/registrar.h>
#include <crypta/lib/python/native_yt/cpp/proto.h>
#include <crypta/lib/native/state/common.h>
#include <crypta/graph/export/lib/proto/messages.pb.h>
#include <crypta/graph/export/lib/native/constants.h>
#include <crypta/graph/export/lib/native/utils.h>
#include <crypta/graph/lib/util/util.h>
#include <crypta/idserv/data/graph.h>
#include <util/generic/hash_set.h>
#include <util/generic/queue.h>
#include <util/generic/map.h>
#include <util/generic/set.h>
#include <util/digest/city.h>
#include <util/random/random.h>

#include <iostream>
#include <tuple>

using NYT::IMapper;
using NYT::IReducer;
using NYT::TTableReader;
using NYT::TTableWriter;

using std::tie;
using google::protobuf::Message;

using TGraphRecord = NCrypta::NGraph::TGraphRecord;
using namespace NCrypta::NGraph;


class TProfileFilterMapper
   : public TStateful<TTimestampState, IMapper<TTableReader<TRecordWithProfile>, TTableWriter<TRecordWithProfile>>> {
public:
    TProfileFilterMapper(const TBuffer& buffer) : TStateful(buffer) {}

    TProfileFilterMapper() : TStateful() {}

    void Do(TTableReader<TRecordWithProfile>* input, TTableWriter<TRecordWithProfile>* output) override;
};

class TFilterReducer
   : public IReducer<TTableReader<TRecordWithProfile>, TTableWriter<TRecordWithProfile>> {
public:
    void Do(TTableReader<TRecordWithProfile>* input, TTableWriter<TRecordWithProfile>* output) override;
};


namespace NV2 {
    using TTimestampState = NCrypta::NGraph::TTimestampState;
    using NYT::TNode;

    class TMapEdgesWithActiveIdentifiers: public IMapper<TTableReader<TNode>, TTableWriter<TEdgeV2>> {
    public:
        TMapEdgesWithActiveIdentifiers(const TBuffer& buffer)
            : State(buffer)
        {
        }

        TMapEdgesWithActiveIdentifiers()
            : State()
        {
        }

        void Do(TTableReader<TNode>* input, TTableWriter<TEdgeV2>* output) override {
            OldestActiveTimestamp = State->GetOldestActiveTimestamp();
            OldestBBActiveTimestamp = State->GetOldestBBActiveTimestamp();

            for (; input->IsValid(); input->Next()) {
                auto& row = input->GetRow();
                TEdgeV2 out;
                switch (static_cast<int>(input->GetTableIndex())) {
                    case 1:    // TV Edge
                        // smart tv has't dates, so set active by hands
                        out.SetIsEdgeActive(true);
                    case 0: {  // V2 Edge
                        CastToEdge(row, out);
                        SetID(out, row, NEdgeV2Fields::ID_1, NEdgeV2Fields::ID_1_TYPE);
                        output->AddRow(out);

                        SetID(out, row, NEdgeV2Fields::ID_2, NEdgeV2Fields::ID_2_TYPE);
                        out.SetFake(true);
                        output->AddRow(out);

                        break;
                    }
                    case 2: {  // Active profiles
                        if (static_cast<time_t>(row["Timestamp"].AsUint64()) < OldestBBActiveTimestamp) {
                            continue;
                        }
                        SetIDValues(out, row[NIdentifierFields::ID].AsString(), row[NIdentifierFields::ID_TYPE].AsString());
                        out.SetIsVertexActive(true);
                        const auto& queriesCount = row["QueriesCount"];
                        const auto& profileSize = row["ProfileSize"];
                        if (!queriesCount.IsNull()) {
                            out.SetQueriesCount(queriesCount.AsUint64());
                        }
                        if (!profileSize.IsNull()) {
                            out.SetProfileSize(profileSize.AsUint64());
                        }
                        output->AddRow(out);
                        break;
                    }
                    default: {  // Identifiers
                        bool withOneDayFilter = false;
                        bool isPrivateID = false;
                        if (row[NIdentifierFields::ID_TYPE].AsString() == YANDEXUID) {

                            /*withOneDayFilter = true;*/ // save active private yandexuids

                            isPrivateID = IsPrivate(row[NIdentifierFields::DATES].AsList(), row[NIdentifierFields::ID].AsString());
                            out.SetIsPrivate(isPrivateID);
                        }
                        ui64 daysCount = 0;
                        bool isVertexActive = ((!row.HasKey(NIdentifierFields::DATES))
                                              || IsDateActive(row[NIdentifierFields::DATES], withOneDayFilter, &daysCount));
                        SetID(out, row, NIdentifierFields::ID, NIdentifierFields::ID_TYPE);
                        out.SetIsVertexActive(isVertexActive);
                        if (daysCount) {
                            out.SetDaysCount(daysCount);
                        }
                        if (isPrivateID || isVertexActive) {
                            output->AddRow(out);
                        }
                        break;
                    }
                }
            }
        }

        void Save(IOutputStream& output) const override {
            State.Save(output);
        }

        void Load(IInputStream& input) override {
            State.Load(input);
        }

    private:
        time_t OldestActiveTimestamp = 0;
        time_t OldestBBActiveTimestamp = 0;
        NNativeYT::TProtoState<TTimestampState> State;

        void SetID(TEdgeV2& edge, const TNode& row, TString keyID, TString keyIDType) {
            SetIDValues(edge, row[keyID].AsString(), row[keyIDType].AsString());
        }

        void SetIDValues(TEdgeV2& edge, TString valueID, TString valueIDType) {
            edge.SetID(valueID);
            edge.SetIDType(valueIDType);
        }

        void CastToEdge(const TNode& row, TEdgeV2& edge) {

            ui64 cryptaID = FromString<ui64>(row[NEdgeV2Fields::CRYPTA_ID].AsString());
            edge.SetCryptaID(cryptaID);
            edge.SetID1(row[NEdgeV2Fields::ID_1].AsString());
            edge.SetID1Type(row[NEdgeV2Fields::ID_1_TYPE].AsString());
            edge.SetID2(row[NEdgeV2Fields::ID_2].AsString());
            edge.SetID2Type(row[NEdgeV2Fields::ID_2_TYPE].AsString());
            edge.SetSourceType(row[NEdgeV2Fields::SOURCE_TYPE].AsString());
            edge.SetLogSource(row[NEdgeV2Fields::LOG_SOURCE].AsString());
            if (row.HasKey(NEdgeV2Fields::DATES)) {
                edge.SetIsEdgeActive(IsDateActive(row[NEdgeV2Fields::DATES]));
            }

            if (row.HasKey(NEdgeV2Fields::INDEVICE)) {
                auto value = row[NEdgeV2Fields::INDEVICE];
                if (value.IsBool()) {
                    edge.SetIndevice(value.AsBool());
                }
            }
            if (row.HasKey(NEdgeV2Fields::SURVIVAL_WEIGHT)) {
                auto value = row[NEdgeV2Fields::SURVIVAL_WEIGHT];
                if (value.IsDouble()) {
                    edge.SetWeight(value.AsDouble());
                }
            }
        }

        bool IsPrivate(const TNode::TListType& dates, const TString& yuid) {
            return ::IsPrivate(dates, yuid);
        }

        bool IsDateActive(const TNode& dates) {
            ui64 daysCount = 0;
            return IsDateActive(dates, false, &daysCount);
        }

        bool IsDateActive(const TNode& dates, bool withOneDayFilter, ui64* daysCount) {
            if (dates.IsList()) {
                return IsDateActive(dates.AsList(), withOneDayFilter, daysCount);
            }
            else if (!dates.IsNull()) {
                ythrow yexception() << "Unknown dates type " << dates.GetType();
            }
            return false;
        }

        bool IsDateActive(const TNode::TListType& dates, bool withOneDayFilter, ui64* daysCount) {
            if (withOneDayFilter && dates.size() < 2) {
                return false;
            }
            bool result = false;
            *daysCount = 0;
            for (const auto& date : dates) {
                time_t timestamp;
                ParseISO8601DateTime(date.AsString().data(), timestamp);
                if (OldestActiveTimestamp < timestamp) {
                    result = true;
                }
                if (OldestActiveTimestamp < timestamp + 14*24*60*60) {
                    (*daysCount)++;
                }
                if (result && (*daysCount) >= 30) {
                    return result;
                }
            }
            return result;
        }
    };

    class TReduceTVEdges: public IReducer<TTableReader<TEdgeV2WithStringCryptaId>, TTableWriter<TEdgeV2WithStringCryptaId>> {
    public:
        void Do(TTableReader<TEdgeV2WithStringCryptaId>* input, TTableWriter<TEdgeV2WithStringCryptaId>* output) override {
            ui64 count = 0;
            for (; input->IsValid(); input->Next()) {
                auto row = input->GetRow();
                count++;

                row.SetID(row.GetID2());
                row.SetIDType(row.GetID2Type());

                output->AddRow(row);

                if (count > MAX_TV_EDGES_PER_NODE) {
                    return;
                }
            }
        }
    };


    class TFilterUsedTVEdges: public IReducer<TTableReader<TEdgeV2WithStringCryptaId>, TTableWriter<TEdgeV2WithStringCryptaId>> {
    public:
        void Do(TTableReader<TEdgeV2WithStringCryptaId>* input, TTableWriter<TEdgeV2WithStringCryptaId>* output) override {
            TEdgeV2WithStringCryptaId edge;
            for (; input->IsValid(); input->Next()) {
                auto& row = input->GetRow();
                if (row.GetID2()) {
                    edge = row;
                } else {
                    return;
                }
            }
            if (edge.GetID2()) {
                edge.SetID(edge.GetID1());
                edge.SetIDType(edge.GetID1Type());
                output->AddRow(edge);
            }
        }
    };

    class TJoinTVEdgesWithCryptaID : public IReducer<TTableReader<TEdgeV2WithStringCryptaId>, TTableWriter<TEdgeV2WithStringCryptaId>> {
    public:
        void Do(TTableReader<TEdgeV2WithStringCryptaId>* input, TTableWriter<TEdgeV2WithStringCryptaId>* output) override {
            ui64 cryptaID = 0;
            TVector<TEdgeV2WithStringCryptaId> edges;
            for (; input->IsValid(); input->Next()) {
                auto& row = input->GetRow();

                if (!row.GetID2()) {
                    cryptaID = FromString<ui64>(row.GetCryptaID());
                } else {

                    edges.push_back(row);
                }
            }
            if (cryptaID) {
                for (TEdgeV2WithStringCryptaId& edge : edges) {
                    edge.SetCryptaID(ToString(cryptaID));
                    edge.SetID(edge.GetID1());
                    edge.SetIDType(edge.GetID1Type());
                    output->AddRow(edge);
                }
            }
        }
    };

    class TReduceEdgesWithActiveIdentifiers: public IReducer<TTableReader<TEdgeV2>, TTableWriter<TEdgeV2>> {
    public:
        void Do(TTableReader<TEdgeV2>* input, TTableWriter<TEdgeV2>* output) override {
            // reduce_by (id, id_type)
            // sort_by (id, id_type, cryptaId)
            ui64 cryptaID = 0;
            bool isVertexActive = false;
            TEdgeV2 vertex;
            TEdgeV2 privateVertex;


            for (bool first = true; input->IsValid(); input->Next(), first = false) {
                auto row = input->GetRow();

                if (first) {
                    vertex.SetID(row.GetID());
                    vertex.SetIDType(row.GetIDType());
                }

                if (!row.GetCryptaID()) {
                    // before edges with cryptaId
                    // identifier
                    isVertexActive |= row.GetIsVertexActive();
                    if (row.GetQueriesCount()) {
                        vertex.SetQueriesCount(row.GetQueriesCount());
                    }
                    if (row.GetProfileSize()) {
                        vertex.SetProfileSize(row.GetProfileSize());
                    }
                    if (row.GetDaysCount()) {
                        vertex.SetDaysCount(row.GetDaysCount());
                    }
                    if (row.GetIsPrivate()) {
                        privateVertex = row;
                    }

                } else {
                    cryptaID = row.GetCryptaID();
                    TEdgeV2 out(row);
                    // vertex is active if it's active itself or is linked by active edge
                    isVertexActive |= row.GetIsEdgeActive();
                    if (isVertexActive) {
                        out.SetIsVertexActive(true);
                    }
                    out.SetQueriesCount(vertex.GetQueriesCount());
                    out.SetProfileSize(vertex.GetProfileSize());
                    out.SetDaysCount(vertex.GetDaysCount());

                    output->AddRow(FillNegativeProperties(out));
                }
            }
            if (cryptaID) {
                if (privateVertex.GetID()) {
                    privateVertex.SetCryptaID(cryptaID);
                    output->AddRow(FillNegativeProperties(privateVertex));
                }

                if (vertex.GetIDType() == MAC) {
                    TEdgeV2 edgeToMd5;
                    TString macExtMd5 = ConvertMacToMD5(vertex.GetID());
                    edgeToMd5.SetCryptaID(cryptaID);
                    edgeToMd5.SetID1(vertex.GetID());
                    edgeToMd5.SetID1Type(vertex.GetIDType());
                    edgeToMd5.SetID2(macExtMd5);
                    edgeToMd5.SetID2Type(MAC_EXT_MD5);

                    edgeToMd5.SetID(edgeToMd5.GetID1());
                    edgeToMd5.SetIDType(edgeToMd5.GetID1Type());
                    edgeToMd5.SetIsVertexActive(isVertexActive);
                    output->AddRow(FillNegativeProperties(edgeToMd5));

                    edgeToMd5.SetID(edgeToMd5.GetID2());
                    edgeToMd5.SetIDType(edgeToMd5.GetID2Type());
                    edgeToMd5.SetFake(true);
                    output->AddRow(FillNegativeProperties(edgeToMd5));

                    if (isVertexActive) {
                        TEdgeV2 vertexMd5;
                        vertexMd5.SetIsVertexActive(true);
                        vertexMd5.SetID(macExtMd5);
                        vertexMd5.SetIDType(MAC_EXT_MD5);
                        vertexMd5.SetCryptaID(cryptaID);
                        output->AddRow(vertexMd5, 1);
                    }
                }

                if (isVertexActive) {
                    vertex.SetCryptaID(cryptaID);
                    vertex.SetIsVertexActive(true);
                    output->AddRow(vertex, 1);
                }

            }
        }
    };

    class TCombine: public IReducer<TTableReader<TEdgeV2>, TTableWriter<TGraphRecord>> {
    public:

        class TInactiveVertexCollector {
        public:
            TInactiveVertexCollector(): VertexDegrees(), ActiveVertices(), SubGraph() {}
            void ConsiderEdge(const TEdgeV2& edge) {
                bool isEdgeActive = IsEdgeActive(edge);
                bool isNode1 = ConsiderVertex(edge.GetID1Type(), edge.GetID1(), isEdgeActive, 1);
                bool isNode2 = ConsiderVertex(edge.GetID2Type(), edge.GetID2(), isEdgeActive, 1);
                if (isNode1 && isNode2) {
                    AddEdge(edge, SubGraph);
                }
                ConsiderVertex(edge.GetIDType(), edge.GetID(), edge.GetIsVertexActive(), 0);
            }

            THashSet<TId> FindInactiveRemovableVertices() {
                THashSet<TId> result;
                TQueue<TGNode*> queue;
                for (const auto& item: VertexDegrees) {
                    auto* node = item.first;
                    if (item.second <= 1 && !ActiveVertices.contains(node)) {
                        queue.push(node);
                    }
                }
                while (queue.size()) {
                    auto* node = queue.front();
                    queue.pop();
                    result.insert(node->Id);
                    for (const auto& edge: node->Edges) {
                        auto& targetNode = (node->Id == (edge->Node1)->Id) ? edge->Node2 : edge->Node1;
                        if (VertexDegrees.contains(targetNode)) {
                            VertexDegrees[targetNode]--;
                            if (VertexDegrees[targetNode] <= 1 && !ActiveVertices.contains(targetNode) && !result.contains(targetNode->Id)) {
                                queue.push(targetNode);
                            }
                        }
                    }
                }

                return result;
            }


        private:
            THashMap<TGNode*, i64> VertexDegrees;
            THashSet<TGNode*> ActiveVertices;
            TGraph SubGraph;

            void SetActive(TGNode* node) {
                ActiveVertices.insert(node);
            }

            void ConsiderVertex(TGNode* node, bool isActive, i64 degreeAddition) {
                VertexDegrees[node] += degreeAddition;
                if (isActive) {
                    SetActive(node);
                }
            }

            bool ConsiderVertex(const TString& type, const TString& id, bool isActive, i64 degreeAddition) {
                if (FilterType(type)) {
                    TGNode* node = GetOrCreateNode(id, type, SubGraph);
                    ConsiderVertex(node, isActive, degreeAddition);
                    return true;
                }
                return false;
            }

            bool FilterType(const TString& type) {
                return NODE_TYPES_FOR_FILTERING.contains(type);
            }

            bool IsEdgeActive(const TEdgeV2& edge) {
                return edge.GetIsEdgeActive() || edge.GetSourceType() == NSourceTypes::SMART_TV;
            }
        };

        void FilterGraph(TGraph& graph, const THashSet<TId>& inactiveVertices) {
            if (!inactiveVertices.size()) {
                return;
            }
            TGraph filteredGraph;
            for (auto& edge: graph.GetEdges()) {
                if (inactiveVertices.contains(edge->Node1->Id) || inactiveVertices.contains(edge->Node2->Id)) {
                    continue;
                }
                auto* node1 = GetOrCreateNode(edge->Node1->Id, filteredGraph);
                auto* node2 = GetOrCreateNode(edge->Node2->Id, filteredGraph);
                auto* resultEdge = filteredGraph.CreateEdge(node1, node2);
                resultEdge->Attributes = edge->Attributes;
                edge->Attributes.clear();
            }
            auto cryptaID = graph.GetId();
            graph.Clear();
            graph.Merge(std::move(filteredGraph));
            graph.SetId(cryptaID);
        }

        void Do(TTableReader<TEdgeV2>* input, TTableWriter<TGraphRecord>* output) override {
            TGraph graph;
            bool isVertexActive = false;
            bool isEdgeActive = false;
            TString filteredID;
            TString filteredIDType;
            TEdgeV2 filteredEdge;
            TInactiveVertexCollector inactiveVertexCollector;
            TPrivateYuidMarker privateYuidMarker;
            TMainYandexuidMarker mainYandexuidMarker;
            graph.SetId(input->GetRow().GetCryptaID());

            for (; input->IsValid(); input->Next()) {
                TEdgeV2 edge = input->GetRow();
                edge = privateYuidMarker.ConsiderAndApply(edge);

                isEdgeActive |= edge.GetIsEdgeActive();
                isVertexActive |= edge.GetIsVertexActive();
                if (edge.GetFake()) {
                    if (!edge.GetID()) {
                        filteredID = edge.GetID2();
                        filteredIDType = edge.GetID2Type();
                        filteredEdge = edge;
                    }
                    continue;
                }

                if ((edge.GetID2() == filteredID) && (edge.GetID2Type() == filteredIDType)) {
                    continue;
                }
                AddEdge(edge, graph);
                inactiveVertexCollector.ConsiderEdge(edge);
            }

            if (!isEdgeActive && !isVertexActive) {
                YieldRejectedGraph("Inactive", graph, output);
                return;
            }
            auto removableVertices = inactiveVertexCollector.FindInactiveRemovableVertices();
            if (IsGraphLarge(graph, removableVertices.size())) {
                // prefilter big graphs
                TGraph emptyGraph;
                emptyGraph.SetId(graph.GetId());
                YieldRejectedGraph("Size limit", emptyGraph, output);
                return;
            }
            FilterGraph(graph, removableVertices);
            mainYandexuidMarker.Update(graph);
            if (!graph.GetNodes().size()) {
                if (filteredID) {
                    if (!AddNode(filteredEdge.GetID1(), filteredEdge.GetID1Type(), graph)) {
                       YieldRejectedGraph("Invalid id", graph, output);
                       return;
                    }
                } else {
                    YieldRejectedGraph("Empty", graph, output);
                    return;
                }
            }
            if (IsGraphLarge(graph)) {
                // filter big graphs
                TGraph emptyGraph;
                emptyGraph.SetId(graph.GetId());
                YieldRejectedGraph("Size limit", emptyGraph, output);
                return;
            }

            TGraphRecord out;
            out.MutableGraph()->MergeFrom(ToProto(graph));
            out.SetCryptaID(graph.GetId());
            output->AddRow(out);
            graph.Clear();
        }

    private:
        void YieldRejectedGraph(TString message, const TGraph& graph, TTableWriter<TGraphRecord>* output) {
            TGraphRecord out;
            out.MutableGraph()->MergeFrom(ToProto(graph));
            out.SetCryptaID(graph.GetId());
            out.SetMessage(message);
            output->AddRow(out, 1);
        }

        bool IsGraphLarge(const TGraph& graph, ui32 removableVerticesSize = 0) {
            return graph.GetNodes().size() > MAX_NODES_COUNT_PER_GRAPH + removableVerticesSize
            || graph.GetEdges().size() > MAX_EDGES_COUNT_PER_GRAPH + removableVerticesSize;
        }
    };

    class TFilterUnchangedGraphs
        : public IReducer<TTableReader<TGraphRecord>, TTableWriter<TGraphRecord>> {
    public:
        void Do(TTableReader<TGraphRecord>* input, TTableWriter<TGraphRecord>* output) override;
    };
}
