#include "utils.h"

#include <util/charset/utf8.h>
#include <util/string/cast.h>
#include <util/generic/algorithm.h>
#include <string>
#include <crypta/lib/native/identifiers/lib/id_types/mac_ext.h>
#include <crypta/lib/native/identifiers/lib/id_types/email.h>
#include <crypta/lib/native/identifiers/lib/id_types/phone.h>

#include <crypta/graph/export/serializers/graph/proto/graph_proto_serializer.h>

using TGraphProtoSerializer = NCrypta::NIS::TGraphProtoSerializer;

TGraphProto ToProto(const TGraph& graph) {
    TGraphProto result;
    TGraphProtoSerializer::Serialize(result, graph);
    return result;
}

TGNode* GetOrCreateNode(const TId& nodeId, TGraph& graph) {
    if (!graph.HasNode(nodeId)) {
        graph.CreateNode(nodeId);
    }
    return graph.FindNode(nodeId);
}

TGNode* GetOrCreateNodeWithoutFiltration(const TString& id, const TString& type, TGraph& graph) {
    const auto nodeId = TId(type, id);
    return GetOrCreateNode(nodeId, graph);
}

TGNode* GetOrCreateNode(const TString& id, const TString& type, TGraph& graph) {
    if (type == NV2::EMAIL) {
        NIdentifiers::TEmail email(id);
        return GetOrCreateNodeWithoutFiltration(email.GetMd5(), type, graph);
    }
    if (type == NV2::PHONE) {
        NIdentifiers::TPhone phone(id);
        return GetOrCreateNodeWithoutFiltration(phone.GetMd5(), type, graph);
    }
    return GetOrCreateNodeWithoutFiltration(id, type, graph);
}

bool ValidID(const TString& id) {
    return IsUtf(id) && id.size() <= 1000;
}

ui64 ConvertToCryptaId(const TString& id) {
    ui64 cryptaId;
    if (TryFromString(id, cryptaId)) {
        return cryptaId;
    }
    return false;
}

bool IsSegmentAudience(ui64 value) {
    return (2e9 <= value) && (value < 3e9);
}

bool HasSegmentsExceptAudience(const yabs::proto::Profile::ProfileItem& item) {
    for (ui64 value : item.Getuint_values()) {
        if (!IsSegmentAudience(value)) {
            return true;
        }
    }
    return false;
}

ui32 GetLastUpdateTime(const yabs::proto::Profile& profile) {
    ui32 lastUpdateTime = 0;

    for (const auto& offer : profile.Getoffers()) {
        if (offer.Getupdate_time() > lastUpdateTime) {
            lastUpdateTime = offer.Getupdate_time();
        }
    }
    for (const auto& item : profile.Getitems()) {
        if (item.Getkeyword_id() == 235 && HasSegmentsExceptAudience(item)) {
            if (item.Getupdate_time() > lastUpdateTime) {
                lastUpdateTime = item.Getupdate_time();
            }
        }
    }
    return lastUpdateTime;
}

ui32 GetQueriesCount(const yabs::proto::Profile& profile) {
    ui32 count = 0;
    for (const auto& query : profile.Getqueries()) {
        if (query.Getquery_text().size() > 0) {
            count++;
        }
    }
    return count;
}

TString ConvertMacToMD5(const TString& strMac) {
    NIdentifiers::TMacExt mac(strMac);
    return mac.GetMd5();
}


namespace NV2 {

    double ComputeNodeWeight(const TEdgeV2& edge) {
        double weight = 0;
        if (edge.GetQueriesCount() > 0) {
            weight += 0.1;
        }
        if (edge.GetProfileSize() > 0) {
            weight += 0.1;
        }
        if (edge.GetDaysCount() > 0 && (edge.GetIsEdgeActive() || edge.GetIsVertexActive())) {

            if (edge.GetIsEdgeActive() || edge.GetIsVertexActive()) {
                weight += 0.05;
            }
        }

        return weight;
    }

    bool AddEdge(const TEdgeV2& edge, TGraph& graph, bool reversed) {
        const TString& id1 = edge.GetID1();
        const TString& id2 = edge.GetID2();
        const TString& id1Type = edge.GetID1Type();
        const TString& id2Type = edge.GetID2Type();

        if (not ValidID(id1) or not ValidID(id2)) {
            return false;
        }

        auto* node1 = GetOrCreateNode(id1, id1Type, graph);
        auto* node2 = GetOrCreateNode(id2, id2Type, graph);
        if (node1 == node2) {
            return false;
        }
        auto queriesCount = edge.GetQueriesCount();
        auto profileSize = edge.GetProfileSize();
        auto daysCount = edge.GetDaysCount();
        auto* node = (edge.GetID() == id1 && edge.GetIDType() == id1Type) ? node1 : node2;;
        if (queriesCount || profileSize) {
            if (queriesCount) {
                node->Attributes[NVertexAttributes::QUERIES_COUNT] = ToString(queriesCount);
            }
            if (profileSize) {
                node->Attributes[NVertexAttributes::PROFILE_SIZE] = ToString(profileSize);
            }
            if (daysCount) {
                node->Attributes[NVertexAttributes::DAYS_COUNT] = ToString(daysCount);
            }
        }
        auto weight = ComputeNodeWeight(edge);
        if (weight > 0) {
            node->Attributes[NVertexAttributes::WEIGHT] = ToString(weight);
        }
        TEdge* resultEdge;
        if (!reversed) {
            resultEdge = graph.CreateEdge(node1, node2);
        } else {
            resultEdge = graph.CreateEdge(node2, node1);
        }

        resultEdge->Attributes[NEdgeAttributes::SOURCE_TYPE] = edge.GetSourceType();
        resultEdge->Attributes[NEdgeAttributes::LOG_SOURCE] = edge.GetLogSource();
        if (edge.GetIdValue()) {
            resultEdge->Attributes[NEdgeAttributes::ID_VALUE] = edge.GetIdValue();
        }
        if (edge.GetPairSource()) {
            resultEdge->Attributes[NEdgeAttributes::PAIR_SOURCE] = edge.GetPairSource();
        }
        if (edge.GetIndevice()) {
            resultEdge->Attributes[NEdgeAttributes::INDEVICE] = "Y";
        }
        if (edge.GetWeight()) {
            resultEdge->Attributes[NEdgeAttributes::WEIGHT] = ToString(edge.GetWeight());
        }
        return true;
    }

    bool AddNode(const TString& id, const TString& idType, TGraph& graph) {
        if (not ValidID(id)) {
            return false;
        }
        GetOrCreateNode(id, idType, graph);
        return true;
    }

    TEdgeV2 FillNegativeProperties(TEdgeV2& row) {
        // just for right sorting (for reducer TCombine)
        if (!row.GetIsPrivate()) {
            row.SetIsNotPrivate(true);
        }
        return row;
    }
}
