#pragma once

#include <crypta/graph/export/lib/native/constants.h>
#include <crypta/graph/export/lib/proto/messages.pb.h>
#include <crypta/idserv/data/graph.h>
#include <crypta/graph/export/proto/graph.pb.h>
#include <crypta/lib/native/identifiers/lib/id_types/yandexuid.h>
#include <yabs/proto/user_profile.pb.h>

#include <util/generic/hash_set.h>

using TGraph = NCrypta::NIS::TGraph;
using TId = NCrypta::NIS::TId;
using TGNode = NCrypta::NIS::TNode;
using TEdge = NCrypta::NIS::TEdge;
using TGraphProto = crypta::idserv::proto::TGraph;
using TEdgeV2 = NCrypta::NGraph::TEdgeV2;
using std::tie;
using std::tuple;

TGraphProto ToProto(const TGraph& graph);

TGNode* GetOrCreateNode(const TId& nodeId, TGraph& graph);

TGNode* GetOrCreateNode(const TString& id, const TString& type, TGraph& graph);

bool ValidID(const TString& id);

ui64 ConvertToCryptaId(const TString& id);

TString ConvertMacToMD5(const TString& mac);

bool IsSegmentAudience(ui64 value);

bool HasSegmentsExceptAudience(const yabs::proto::Profile::ProfileItem& item);

// It's not a very honest last update time.
// It observes offers and items with special views only.
ui32 GetLastUpdateTime (const yabs::proto::Profile& profile);

ui32 GetQueriesCount(const yabs::proto::Profile& profile);



namespace NV2 {
    using TGraph = NCrypta::NIS::TGraph;
    using TId = NCrypta::NIS::TId;
    using TEdgeV2 = NCrypta::NGraph::TEdgeV2;
    using NIdentifiers::TYandexuid;
    using std::optional;

    static bool IsYandexuid(const TString& type) {
        return type == YANDEXUID;
    }

    bool AddEdge(const TEdgeV2& edge, TGraph& graph, bool reversed = false);

    bool AddNode(const TString& id, const TString& idType, TGraph& graph);

    class TPrivateYuidMarker {
        // Collect private yuids. Set PRIVATE_YANDEXUID type for them.
    public:
        TPrivateYuidMarker()
            : PrivateYuids()
        {
        }
        TEdgeV2 ConsiderAndApply(const TEdgeV2& edge) {
            // It's important to consider private edge firstly;
            if (edge.GetIsPrivate()) {
                if (!IsYandexuid(edge.GetIDType())) {
                    ythrow yexception() << "Vertex (" << edge.GetID() << ", " << edge.GetIDType() << ") cannot be private.";
                }
                PrivateYuids.insert(edge.GetID());
            }
            return Apply(edge);
        }

        TEdgeV2 Apply(const TEdgeV2& edge) {
            TEdgeV2 out(edge);
            if (IsYandexuid(edge.GetID1Type())) {
                out.SetID1Type(GetType(edge.GetID1()));
            }
            if (IsYandexuid(edge.GetID2Type())) {
                out.SetID2Type(GetType(edge.GetID2()));
            }
            return out;
        }

    private:
        THashSet<TString> PrivateYuids;
        TString GetType(const TString& yuid) {
            return !PrivateYuids.contains(yuid) ? YANDEXUID : PRIVATE_YANDEXUID;
        }
    };

    class TMainYandexuidMarker {
    public:
        void Update(TGraph& graph) {
            optional<TId> selected{};
            ui64 timestamp{};
            for (auto node : graph.GetNodes()) {
                if (!IsYandexuid(node->Id.Type)) {
                    continue;
                }

                auto yandexuid = TYandexuid(node->Id.Value);
                if (!yandexuid.IsValid()) {
                    continue;
                }

                if (auto candidateTimestamp = yandexuid.GetTimestamp(); !selected || (candidateTimestamp < timestamp)) {
                    selected = node->Id;
                    timestamp = candidateTimestamp;
                }
            }
            if (selected) {
                graph.GetAttributes()["main_yandexuid"] = selected.value().Value;
            }
        }
    };

    TEdgeV2 FillNegativeProperties(TEdgeV2& row);
}
