#include "graph_proto_serializer.h"

#include <crypta/lib/native/proto_serializer/proto_serializer.h>

using namespace NCrypta::NIS;

TString NCrypta::NIS::TGraphProtoSerializer::Serialize(const TGraph& graph) {
    TGraphProto proto;

    Serialize(proto, graph);

    return proto.SerializeAsString();
}

void TGraphProtoSerializer::Serialize(TGraphProtoSerializer::TGraphProto& proto, const TGraph& graph) {
    for (const auto& node : graph.GetNodes()) {
        auto nodeProto = proto.MutableNodes()->Add();

        nodeProto->SetType(node->Id.Type);
        nodeProto->SetId(node->Id.Value);

        SerializeAttributes(*nodeProto->MutableAttributes(), node->Attributes);
    }

    for (const auto& edge : graph.GetEdges()) {
        auto edgeProto = proto.MutableEdges()->Add();

        edgeProto->SetNode1(static_cast<ui32>(graph.IndexOf(edge->Node1)));
        edgeProto->SetNode2(static_cast<ui32>(graph.IndexOf(edge->Node2)));

        SerializeAttributes(*edgeProto->MutableAttributes(), edge->Attributes);
    }

    proto.SetCryptaId(graph.GetId());
    SerializeAttributes(*proto.MutableAttributes(), graph.GetAttributes());
}

TGraph TGraphProtoSerializer::Deserialize(const TStringBuf& protoStr) {
    Y_ENSURE(!protoStr.empty(), "Serialized graph is empty");

    TGraph graph;
    Deserialize(graph, NProtoSerializer::CreateFromString<TGraphProto>(protoStr));
    return graph;
}

void TGraphProtoSerializer::Deserialize(TGraph& graph, const TGraphProto& proto) {
    graph.Clear();

    for (const auto& nodeProto : proto.GetNodes()) {
        auto* node = graph.CreateNode(nodeProto.GetType(), nodeProto.GetId());
        DeserializeAttributes(node->Attributes, nodeProto.attributes());
    }

    for (const auto& edgeProto : proto.GetEdges()) {
        auto* edge = graph.CreateEdge(edgeProto.GetNode1(), edgeProto.GetNode2());
        DeserializeAttributes(edge->Attributes, edgeProto.attributes());
    }

    graph.SetId(proto.GetCryptaId());
    DeserializeAttributes(graph.GetAttributes(), proto.attributes());
}

void TGraphProtoSerializer::SerializeAttributes(TGraphProtoSerializer::TAttributesProto& proto, const TAttributes& attributes) {
    for (const auto& kvp : attributes) {
        auto attr = proto.Add();
        attr->set_name(kvp.first);
        attr->set_value(kvp.second);
    }
}

void TGraphProtoSerializer::DeserializeAttributes(TAttributes& attributes, const TGraphProtoSerializer::TAttributesProto& proto) {
    for (const auto& attr : proto) {
        attributes[attr.name()] = attr.value();
    }
}
