#include <sys/types.h>
#include <sys/socket.h>
#include <linux/if_packet.h>
#include <cstring>
#include <util/generic/yexception.h>
#include "buffers.h"

TCaptureBuffers::TCaptureBuffers(std::unique_ptr<TInterfaceBindedSocket> socket, TBufferOptions options)
    : Socket(std::move(socket))
    , Options(std::move(options))
{
    try {
        SetPacketVersion();
        SetFanOutGroup();
        SetupRingBuffers();
    } catch (...) {
        CleanUp();
        throw;
    }
}

void TCaptureBuffers::SetPacketVersion() {
    int version = TPACKET_V3;
    auto result = setsockopt(fd(), SOL_PACKET, PACKET_VERSION, &version, sizeof(version));
    Y_ENSURE(result != -1, "TCaptureBuffers::SetPacketVersion setsockopt PACKET_VERSION failed");
}

void TCaptureBuffers::SetFanOutGroup() {
    if (Options.FanoutId == -1) {
        return;
    }
    int fanout = (Options.FanoutId | (Options.FanoutType << 16));
    auto result = setsockopt(fd(), SOL_PACKET, PACKET_FANOUT, &fanout, sizeof(fanout));
    Y_ENSURE(result != -1, "TCaptureBuffers::SetFanOutGroup setsockopt PACKET_FANOUT failed");
}

void TCaptureBuffers::CleanUp() {
    delete[] Blocks;

    if (BlocksMemory != MAP_FAILED) {
        munmap(BlocksMemory, Options.BlockSize * Options.BlockNum);
    }
}

TCaptureBuffers::~TCaptureBuffers() {
    CleanUp();
}

void TCaptureBuffers::SetupRingBuffers() {
    tpacket_req3 request;
    memset(&request, 0, sizeof(request));

    request.tp_block_size = Options.BlockSize;
    request.tp_frame_size = Options.FrameSize;
    request.tp_block_nr = Options.BlockNum;
    request.tp_frame_nr = (Options.BlockSize * Options.BlockNum) / Options.FrameSize;
    request.tp_retire_blk_tov = Options.WaitForDataMs;
    request.tp_feature_req_word = TP_FT_REQ_FILL_RXHASH;

    auto result = setsockopt(Socket->fd(), SOL_PACKET, PACKET_RX_RING, (void*)&request, sizeof(request));
    Y_ENSURE(result != -1, "TCaptureBuffers::SetupRingBuffers setsockopt PACKET_RX_RING failed");

    BlocksMemory = (ui8*)mmap(NULL, request.tp_block_size * request.tp_block_nr, PROT_READ | PROT_WRITE, MAP_SHARED | MAP_LOCKED, Socket->fd(), 0);
    Y_ENSURE(BlocksMemory != MAP_FAILED, "TCaptureBuffers::SetupRingBuffers mmap failed");

    Blocks = new iovec[request.tp_block_nr];
    for (ui32 i = 0; i < request.tp_block_nr; ++i) {
        Blocks[i].iov_base = BlocksMemory + (i * request.tp_block_size);
        Blocks[i].iov_len = request.tp_block_size;
    }
}

int TCaptureBuffers::fd() const {
    return Socket->fd();
}

TBlockData* TCaptureBuffers::GetBlock(ui32 index) const {
    return (TBlockData*)Blocks[index].iov_base;
}

void TCaptureBuffers::FlushBlock(ui32 index) {
    auto block = (TBlockData*)Blocks[index].iov_base;
    block->h1.block_status = TP_STATUS_KERNEL;
}

bool TCaptureBuffers::IsBlockWithData(TBlockData* block) {
    return (block->h1.block_status & TP_STATUS_USER) != 0;
}
