import datetime
import os

from cached_property import cached_property
from crypta.lib.python.yql_runner.task import YQLRunnerTask
import crypta.lib.python.bt.conf.conf as conf

from crypta.graph.soup.config.python import (
    SOURCE_TYPE,
    LOG_SOURCE,
    EDGE_TYPE,
    ID_TYPE,
)

from crypta.lib.python.bt.workflow import (
    IndependentTask,
    Parameter,
)

from crypta.lib.python.bt.workflow.targets.table import (
    HasAttribute,
)

from crypta.lib.python.bt.paths import (
    Table,
)

import logging
logger = logging.getLogger(__name__)


class Paths(object):
    FP_OUTPUT_DIR = "//home/crypta/{subpath}/state/fingerprint"
    FP_SOUP_PATH = "//home/crypta/{subpath}/state/graph/v2/soup/{table_name}"

    edge_type = EDGE_TYPE.get_edge_type(ID_TYPE.MM_DEVICE_ID,
                                        ID_TYPE.MM_DEVICE_ID,
                                        SOURCE_TYPE.APP_METRICA_IOS,
                                        LOG_SOURCE.FINGERPRINT)

    # TODO: CRYPTA-13831 Add ssp_ios to SOURCE_TYPE
    # ssp_edge_type = EDGE_TYPE.get_edge_type(ID_TYPE.SSP_USER_ID,
    #                                     ID_TYPE.MM_DEVICE_ID,
    #                                     SOURCE_TYPE.SSP_IOS,
    #                                     LOG_SOURCE.FINGERPRINT)

    def __init__(self, task):
        self.task = task

    def metrika_mobile_log(self, date):
        return os.path.join(conf.proto.MetrikaMobileLog, date)

    def metrika_location_log(self, date):
        return os.path.join(conf.proto.MetrikaLocationLog, date)

    def ssp_rtb_log(self, date):
        return os.path.join(conf.proto.RtbLog, date)

    @property
    def subpath(self):
        crypta_env = str(self.task.crypta_env)
        env = os.environ.get("CRYPTA_ENVIRONMENT")
        if env is not None:
            crypta_env = str(env)
        crypta_env = crypta_env.lower()
        subpath = crypta_env
        if crypta_env == "develop":
            subpath = "team/{}/develop".format(os.getlogin())
        return subpath

    def soup_table_name(self, edge_type):
        return "{id1_type}_{id2_type}_{source_type}_{log_source}".format(
            id1_type=edge_type.Id1Type.Name,
            id2_type=edge_type.Id2Type.Name,
            source_type=edge_type.SourceType.Name,
            log_source=edge_type.LogSource.Name,
        )

    @cached_property
    def _base_output_dir(self):
        return self.FP_OUTPUT_DIR.format(subpath=self.subpath)

    @cached_property
    def app_output_dir(self):
        return os.path.join(self._base_output_dir, "appmetrica")

    @cached_property
    def ssp_output_dir(self):
        return os.path.join(self._base_output_dir, "ssp")

    def _output(self, type, name, date, output_dir):
        _dir = os.path.join(output_dir, type) if type is not None else output_dir
        _dir = os.path.join(_dir, name)
        if date is None:
            return _dir
        return os.path.join(_dir, date)

    def app_output_pairs(self, type, date=None):
        return self._output(type, "pairs", date, self.app_output_dir)

    def app_output_ids(self, type, date=None):
        return self._output(type, "ids", date, self.app_output_dir)

    @property
    def app_output_all_pairs(self):
        return os.path.join(self.app_output_dir, "all_pairs")

    @property
    def app_output_all_stats(self):
        return os.path.join(self.app_output_dir, "all_stats")

    def ssp_output_pairs(self, date=None):
        return self._output(None, "pairs", date, self.ssp_output_dir)

    def ssp_output_stats(self, date=None):
        return self._output(None, "stats", date, self.ssp_output_dir)

    @property
    def ssp_output_all_pairs(self):
        return os.path.join(self.ssp_output_dir, "all_pairs")

    @property
    def ssp_output_all_stats(self):
        return os.path.join(self.ssp_output_dir, "all_stats")

    @property
    def ssp_output_soup(self):
        table_name = "{id1_type}_{id2_type}_{source_type}_{log_source}".format(
            id1_type="ssp_user_id",
            id2_type="mm_device_id",
            source_type="ssp-app-metrica-ios",
            log_source="fingerprint",
        )
        return self.FP_SOUP_PATH.format(subpath=self.subpath, table_name=table_name)

        # return self.FP_SOUP_PATH.format(subpath=self.subpath,
        #                                 table_name=self.soup_table_name(self.ssp_edge_type))

    @property
    def app_output_soup(self):
        return self.FP_SOUP_PATH.format(subpath=self.subpath,
                                        table_name=self.soup_table_name(self.edge_type))

    @property
    def expiration_duration(self):
        return int(conf.proto.ExpirationTimeInDays)

    def set_expiration_time(self, path, date=None):
        days = self.expiration_duration
        if days < 0:
            return
        if date is None:
            date = str(self.task.date)
        dt = datetime.datetime.strptime(date, "%Y-%m-%d") + datetime.timedelta(days=days)
        self.task.yt.set("{path}/@expiration_time".format(path=path), "{dt:%Y-%m-%d} 00:00:00.0+00:00".format(dt=dt))


def convert_date_to_ts(date):
    return int(datetime.datetime.strptime(date, "%Y-%m-%d").strftime("%s"))


class FingerprintQuery(YQLRunnerTask, IndependentTask):

    """ Base fp query class """
    MOBILE = "mobile"
    MOBILE_IP = "mobile_ip"
    LOCATION = "location"
    IDENTITY = "identity"

    STATS_ATTRIBUTE = "stats"
    DATE_ATTRIBUTE = "generate_date"

    @cached_property
    def date(self):
        return conf.proto.Date

    @cached_property
    def ts_date(self):
        return convert_date_to_ts(self.date) - convert_date_to_ts("1970-01-01")

    @cached_property
    def paths(self):
        return Paths(self)

    def get_context_data(self, **kwargs):
        context = super(FingerprintQuery, self).get_context_data(**kwargs)
        context.update(
            date=self.date, fp_dir=self.paths._base_output_dir
        )
        return context

    def get_libs(self):
        return []

    def stats_attr_path(self, path):
        return "{}/@{}".format(path, self.STATS_ATTRIBUTE)

    def date_attr_path(self, path):
        return "{}/@{}".format(path, self.DATE_ATTRIBUTE)

    def run(self, *args, **kwargs):
        logger.info('Date %s', self.date)
        super(FingerprintQuery, self).run(*args, **kwargs)


class BaseFPCollector(FingerprintQuery):
    output = Parameter()

    @property
    def outputs(self):
        return [self.output]

    def targets(self):
        for out in self.outputs:
            yield Table(self.yt, str(out)).be.exists()

    def run(self, *args, **kwargs):
        super(BaseFPCollector, self).run(*args, **kwargs)

        for out in self.outputs:
            self.paths.set_expiration_time(str(out))


class AbstractCombinePairs(FingerprintQuery):
    @property
    def input_dirs(self):
        raise NotImplementedError()

    @property
    def input_paths(self):
        raise NotImplementedError()

    @property
    def output_all_pairs(self):
        raise NotImplementedError()

    @property
    def output_all_stats(self):
        raise NotImplementedError()

    def save_stats(self):
        return False

    @property
    def _input_ranges(self):
        result = []
        for _dir in filter(self.yt.exists, self.input_dirs):
            names = self.yt.list(_dir)
            if len(names) > 0:
                _from = min(names)
                _to = max(names)
                result.append((_dir, _from, _to))
        return result

    @property
    def _input_paths(self):
        return list(filter(self.yt.exists, self.input_paths))

    @property
    def _jinja_extensions(self):
        return super(AbstractCombinePairs, self)._jinja_extensions + [
            "jinja2.ext.do", ]

    @property
    def query_template(self):
        return "combine.sql.j2"

    def get_range(self, _dir):
        names = self.yt.list(_dir)
        return (_dir, min(names), max(names))

    def get_context_data(self, **kwargs):
        context = super(AbstractCombinePairs, self).get_context_data(
            **kwargs)

        context.update(
            output_all_pairs=self.output_all_pairs,
            output_stats=self._output_stats,
            input_ranges=self._input_ranges,
            input_paths=self._input_paths,
            with_cryptaid=False,
            with_sspid=False,
            soup_filter='base_soup_filter.sql.j2',
        )
        return context

    def run(self, **kwargs):
        with self.yt.TempTable(prefix='stats_') as self._output_stats:
            super(AbstractCombinePairs, self).run(**kwargs)
            stats = self.yt.read_table(self._output_stats).next()
            self.yt.set(self.stats_attr_path(self.output_all_pairs), stats)
            self.yt.set(self.date_attr_path(self.output_all_pairs), self.date)

            if self.save_stats():
                path = self.output_all_stats
                if self.yt.exists(path):
                    path = "<append=true>{}".format(path)
                self.yt.write_table(path, [stats])


class HasAttributeNotLessThan(HasAttribute):
    def __init__(self, yt, table, attribute, value):
        self.notless_value = value
        super(HasAttributeNotLessThan, self).__init__(yt, table, attribute)

    def satisfied(self):
        if not super(HasAttributeNotLessThan, self).satisfied():
            return False
        attr_value = self.yt.get("{}/@{}".format(self._table, self._attribute))
        if attr_value < self.notless_value:
            logger.debug('Table %s has %s = %s, expected >= %s',
                         self._table, self._attribute, attr_value, self._value)
            return False
        return True


class FingerprintQueryWithDateTargets(FingerprintQuery):

    @property
    def target_outputs(self):
        raise NotImplementedError()

    def targets(self):
        for out in self.target_outputs:
            yield HasAttributeNotLessThan(
                self.yt, out, self.DATE_ATTRIBUTE,
                str(self.date)
            )

    def run(self, **kwargs):
        for out in self.target_outputs:
            if self.yt.exists(out):
                self.yt.remove(out)
        super(FingerprintQueryWithDateTargets, self).run(**kwargs)
        for out in self.target_outputs:
            self.yt.set(self.date_attr_path(out), self.date)
