from crypta.graph.fingerprint.lib import (
    FingerprintQuery,
    BaseFPCollector,
    AbstractCombinePairs,
    HasAttributeNotLessThan,
    FingerprintQueryWithDateTargets,
)
from crypta.lib.python.bt.workflow import (
    Parameter,
)

from crypta.lib.python.bt.paths import (
    Table,
)

import crypta.lib.python.bt.conf.conf as conf

import logging
logger = logging.getLogger(__name__)


class CollectMobileStaticFP(BaseFPCollector):
    MOBILE_STATIC = "mobile_static"

    @property
    def output(self):
        return self.paths.app_output_ids(self.MOBILE_STATIC, self.date)

    @property
    def query_template(self):
        return "mobile_static_fp.sql.j2"

    def get_context_data(self, **kwargs):
        context = super(CollectMobileStaticFP, self).get_context_data(**kwargs)
        output_path = str(self.output)

        context.update(
            input_path=str(self.paths.metrika_mobile_log(self.date)),
            output_path=str(output_path),
            output_stats=str(self.output_stats)
        )
        return context

    def run(self, **kwargs):
        with self.yt.TempTable(prefix='stats_') as self.output_stats:
            super(CollectMobileStaticFP, self).run(**kwargs)
            self.yt.set(self.stats_attr_path(self.output), self.yt.read_table(self.output_stats).next())


class CollectLocationFP(BaseFPCollector):
    @property
    def query_template(self):
        return "location_fp.sql.j2"

    def get_context_data(self, **kwargs):
        context = super(CollectLocationFP, self).get_context_data(**kwargs)
        output_path = str(self.output)

        context.update(
            input_path=str(self.paths.metrika_location_log(self.date)),
            output_path=output_path,
            location_p=8,
            timestamp_p=100
        )
        return context


class CollectMobileFP(BaseFPCollector):

    @property
    def query_template(self):
        return "mobile_fp.sql.j2"

    def get_context_data(self, **kwargs):
        context = super(CollectMobileFP, self).get_context_data(**kwargs)
        output_path = str(self.output)

        context.update(
            input_path=str(self.paths.metrika_mobile_log(self.date)),
            output_path=output_path,
            timestamp_p=1000
        )
        return context


class CollectIdentityLightFP(BaseFPCollector):

    @property
    def query_template(self):
        return "identity_fp.sql.j2"

    def get_context_data(self, **kwargs):
        context = super(CollectIdentityLightFP, self).get_context_data(**kwargs)
        output_path = str(self.output)

        context.update(
            input_path=str(self.paths.metrika_mobile_log(self.date)),
            output_path=output_path,
        )
        return context


class CollectMobileIPFP(BaseFPCollector):

    @property
    def query_template(self):
        return "mobile_ip_fp.sql.j2"

    def get_context_data(self, **kwargs):
        context = super(CollectMobileIPFP, self).get_context_data(**kwargs)
        output_path = str(self.output)

        context.update(
            input_path=str(self.paths.metrika_mobile_log(self.date)),
            output_path=output_path,
        )
        return context


class CollectIDsStats(FingerprintQuery):
    ids = Parameter()

    @property
    def stats_attribute_path(self):
        return self.stats_attr_path(self.ids)

    def targets(self):
        yield Table(self.yt, str(self.stats_attribute_path)).be.exists()

    @property
    def query_template(self):
        return "stats.sql.j2"

    def get_context_data(self, **kwargs):
        context = super(CollectIDsStats, self).get_context_data(**kwargs)
        input_path = str(self.ids)
        output_stats = str(self.output_stats)

        context.update(
            input_path=input_path,
            output_stats=output_stats
        )
        return context

    def run(self, **kwargs):
        with self.yt.TempTable(prefix='stats_') as self.output_stats:
            super(CollectIDsStats, self).run(**kwargs)
            self.yt.set(self.stats_attribute_path, self.yt.read_table(self.output_stats).next())


class CollectMobileFPWithStats(CollectIDsStats):

    def requires(self):
        yield CollectMobileFP(output=self.ids)


class CollectLocationFPWithStats(CollectIDsStats):

    def requires(self):
        yield CollectLocationFP(output=self.ids)


class AbstractCollectPairsFromFP(FingerprintQuery):
    @property
    def fp_type(self):
        raise NotImplementedError()

    @property
    def input_ids(self):
        return self.paths.app_output_ids(self.fp_type, self.date)

    @property
    def output_pairs(self):
        return self.paths.app_output_pairs(self.fp_type, self.date)

    def targets(self):
        yield Table(self.yt, str(self.output_pairs)).be.exists()

    @property
    def query_template(self):
        return "pairs.sql.j2"

    @property
    def max_fp_size(self):
        return 15

    def get_context_data(self, **kwargs):
        context = super(AbstractCollectPairsFromFP, self).get_context_data(**kwargs)
        input_path = str(self.input_ids)
        output_path = str(self.output_pairs)
        output_stats = str(self.output_stats)

        context.update(
            input_path=input_path,
            output_path=output_path,
            output_stats=output_stats,
            max_fp_size=self.max_fp_size,
            fp_type=self.fp_type,
            ts=self.ts_date
        )
        return context

    def run(self, **kwargs):
        with self.yt.TempTable(prefix='stats_') as self.output_stats:
            super(AbstractCollectPairsFromFP, self).run(**kwargs)
            stats = list(self.yt.read_table(self.output_stats))
            self.yt.set(self.stats_attr_path(self.output_pairs), stats)
            logger.info(stats)
        self.paths.set_expiration_time(self.output_pairs)


class CollectPairsFromMobileFP(AbstractCollectPairsFromFP):
    @property
    def fp_type(self):
        return self.MOBILE

    def requires(self):
        yield CollectMobileFP(output=self.input_ids)


class CollectPairsFromIdentityFP(AbstractCollectPairsFromFP):
    @property
    def fp_type(self):
        return self.IDENTITY

    @property
    def max_fp_size(self):
        return 15

    def requires(self):
        yield CollectIdentityLightFP(output=self.input_ids)


class CollectPairsFromMobileIPFP(AbstractCollectPairsFromFP):
    @property
    def fp_type(self):
        return self.MOBILE_IP

    @property
    def max_fp_size(self):
        return 5

    def requires(self):
        yield CollectMobileIPFP(output=self.input_ids)


class CollectPairsFromLocationFP(AbstractCollectPairsFromFP):
    @property
    def fp_type(self):
        return self.LOCATION

    @property
    def max_fp_size(self):
        return 7

    def requires(self):
        yield CollectLocationFP(output=self.input_ids)


class CollectPairsFromCustomFp(AbstractCollectPairsFromFP):
    input_ids = Parameter()
    output_pairs = Parameter()


class AppCombinePairs(AbstractCombinePairs):
    @property
    def output_all_pairs(self):
        return self.paths.app_output_all_pairs

    @property
    def output_all_stats(self):
        return self.paths.app_output_all_stats

    def get_context_data(self, **kwargs):
        context = super(AppCombinePairs, self).get_context_data(
            **kwargs)

        context.update(
            cnt_threshold=conf.proto.CountThreshold,
            fp_types_cnt_threshold=conf.proto.FpTypesCountThreshold,
            soup_filter='app_soup_filter.sql.j2',
            with_cryptaid=True,
        )
        return context


class RecombineAllPairs(AppCombinePairs):
    @property
    def input_dirs(self):
        return [
            self.paths.app_output_pairs(self.LOCATION),
            self.paths.app_output_pairs(self.MOBILE),
            self.paths.app_output_pairs(self.MOBILE_IP),
            self.paths.app_output_pairs(self.IDENTITY)
        ]

    @property
    def input_paths(self):
        return []

    def run(self, **kwargs):
        if self.yt.exists(self.paths.app_output_all_pairs):
            self.yt.remove(self.paths.app_output_all_pairs)
        if self.yt.exists(self.paths.app_output_all_stats):
            self.yt.remove(self.paths.app_output_all_stats)

        super(RecombineAllPairs, self).run(**kwargs)


class MergeDailyFPWithCountStorage(AppCombinePairs):

    def requires(self):
        yield CollectPairsFromLocationFP()
        yield CollectPairsFromMobileFP()
        yield CollectPairsFromMobileIPFP()
        yield CollectPairsFromIdentityFP()

    def targets(self):
        yield HasAttributeNotLessThan(
            self.yt, self.paths.app_output_all_pairs, self.DATE_ATTRIBUTE, str(self.date)
        )

    @property
    def input_dirs(self):
        return []

    def save_stats(self):
        return True

    @property
    def input_paths(self):
        return [
            self.paths.app_output_all_pairs,
            self.paths.app_output_pairs(self.LOCATION, self.date),
            self.paths.app_output_pairs(self.MOBILE_IP, self.date),
            self.paths.app_output_pairs(self.MOBILE, self.date),
            self.paths.app_output_pairs(self.IDENTITY, self.date)
        ]


class UpdateSoupEdges(FingerprintQueryWithDateTargets):

    def requires(self):
        yield MergeDailyFPWithCountStorage()

    @property
    def output(self):
        return self.paths.app_output_soup

    @property
    def target_outputs(self):
        return [self.output]

    @property
    def query_template(self):
        return "soup.sql.j2"

    def get_context_data(self, **kwargs):
        context = super(UpdateSoupEdges, self).get_context_data(**kwargs)

        context.update(
            all_pairs=self.paths.app_output_all_pairs,
            output_soup_edges=self.output,
            soup_weight=conf.proto.WeightForSoupEdges,
        )
        return context
