from crypta.graph.fingerprint.lib import (
    BaseFPCollector,
    AbstractCombinePairs,
    HasAttributeNotLessThan,
    FingerprintQueryWithDateTargets,
)

import crypta.lib.python.bt.conf.conf as conf

import logging
logger = logging.getLogger(__name__)


class CollectPairs(BaseFPCollector):

    @property
    def output(self):
        return self.paths.ssp_output_pairs(self.date)

    @property
    def output_stats(self):
        return self.paths.ssp_output_stats(self.date)

    @property
    def outputs(self):
        return [self.output, self.output_stats]

    @property
    def query_template(self):
        return "ssp_app_fp.sql.j2"

    def get_context_data(self, **kwargs):
        context = super(CollectPairs, self).get_context_data(**kwargs)

        context.update(
            date=self.date,
            rtb_path=self.paths.ssp_rtb_log(self.date),
            appmetrica_path=self.paths.metrika_mobile_log(self.date),
            output_pairs=self.output,
            output_stats=self.output_stats,
            params=[
                ("13483990", ["ip", "os_v", "lang"]),                     # CheetahMobile
                ("17298340", ["ip.0", "os_v", "size"]),                   # Google
                ("40429569", ["ip", "os_v", "size", "lang", "op_name"]),  # PubNative
                ("57829915", ["ip", "os_v", "lang"]),                     # TikTok
                ("8739185", ["ip", "os_v", "lang", "op_id"]),             # MoPub
            ],
            device_threshold=6,
            ts=self.ts_date,
        )
        return context

    def run(self, **kwargs):
        super(CollectPairs, self).run(**kwargs)
        self.yt.set(self.stats_attr_path(self.output),
                    list(self.yt.read_table(self.output_stats)))


class SspCombinePairs(AbstractCombinePairs):
    @property
    def output_all_pairs(self):
        return self.paths.ssp_output_all_pairs

    @property
    def output_all_stats(self):
        return self.paths.ssp_output_all_stats

    def get_context_data(self, **kwargs):
        context = super(SspCombinePairs, self).get_context_data(
            **kwargs)

        context.update(
            cnt_threshold=conf.proto.CountThreshold,
            soup_filter='ssp_soup_filter.sql.j2',
            with_sspid=True,
        )
        return context


class RecombineAllPairs(SspCombinePairs):
    @property
    def input_dirs(self):
        return [
            self.paths.ssp_output_pairs(),
        ]

    @property
    def input_paths(self):
        return []

    def run(self, **kwargs):
        if self.yt.exists(self.paths.ssp_output_all_pairs):
            self.yt.remove(self.paths.ssp_output_all_pairs)
        if self.yt.exists(self.paths.ssp_output_all_stats):
            self.yt.remove(self.paths.ssp_output_all_stats)

        super(RecombineAllPairs, self).run(**kwargs)


class MergeDailyFPWithCountStorage(SspCombinePairs):

    def requires(self):
        yield CollectPairs()

    def targets(self):
        yield HasAttributeNotLessThan(
            self.yt, self.paths.ssp_output_all_pairs, self.DATE_ATTRIBUTE, str(self.date)
        )

    @property
    def input_dirs(self):
        return []

    def save_stats(self):
        return True

    @property
    def input_paths(self):
        return [
            self.output_all_pairs,
            self.paths.ssp_output_pairs(self.date)
        ]


class UpdateSoupEdges(FingerprintQueryWithDateTargets):

    def requires(self):
        yield MergeDailyFPWithCountStorage()

    @property
    def output(self):
        return self.paths.ssp_output_soup

    @property
    def target_outputs(self):
        return [self.output]

    @property
    def query_template(self):
        return "ssp_soup.sql.j2"

    def get_context_data(self, **kwargs):
        context = super(UpdateSoupEdges, self).get_context_data(**kwargs)

        context.update(
            all_pairs=self.paths.ssp_output_all_pairs,
            output_soup_edges=self.output,
            soup_weight=0.01,
        )
        return context
