import datetime

from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import consts
from crypta.lib.python.spine.juggler.juggler_check_generator import JugglerCheckGenerator
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.lib.python.spine.solomon import (
    solomon_alert_utils,
    solomon_check_generator,
)
from sandbox.projects.crypta.graph import fpc


def get_registry():
    juggler = JugglerCheckGenerator(
        host="crypta-fpc",
        tags=["crypta-fpc", "crypta-idfy-telegram-alert"],
        child_group_type=consts.GroupType.host,
        child_group="crypta-fpc-solomon",
    )

    sandbox = sandbox_scheduler.create_default_generator(juggler, ["GRAPH"])

    solomon = solomon_check_generator.SolomonCheckGenerator(
        juggler,
        solomon_alert_utils.AlertCreator(
            project_id="crypta_graph",
            selectors={"project": "crypta", "cluster": "fpc"},
        ),
    )

    solomon.get_sensor({"sensor": "fpc_max_component_size", "service": "fpc"}).create_threshold_check(
        aggregation=alert_pb2.MAX,
        predicate=alert_pb2.GT,
        threshold=4096,
        period=datetime.timedelta(hours=4),
        juggler_service="CryptaGraphFpc_max_component_size",
        description="Max component size == {{ pointValue }}",
    )

    def create_scheduler(
        mode,
        kill_timeout=datetime.timedelta(hours=4),
        export_to_is=False,
        schedule_interval=datetime.timedelta(minutes=20),
        schedule_daily_start_time=None,
        crit_time=datetime.timedelta(hours=12),
    ):
        schedule_interval = schedule_interval if schedule_daily_start_time is None else None

        sb = sandbox.create_scheduler(
            fpc.CryptaGraphFpc,
            schedule_interval=schedule_interval,
            kill_timeout=kill_timeout,
            schedule_daily_start_time=schedule_daily_start_time,
            extra_params={
                "mode": mode,
                "export": export_to_is,
                "pool": "crypta_rt_graph",
            },
        )
        if crit_time is not None:
            sb.check(
                crit_time=crit_time,
                juggler_service=fpc.CryptaGraphFpc.get_juggler_service_for_mode(mode, export_to_is),
            ).add_yt_dependencies(
                YtProxy.Group.offline,
            )

    create_scheduler(
        mode="calculate_sources",
        kill_timeout=datetime.timedelta(minutes=30),
        schedule_interval=datetime.timedelta(hours=1),
    )
    create_scheduler(
        mode="get_fpc_via_cookie_sync",
    )
    create_scheduler(
        mode="get_fpc_via_tls",
    )
    create_scheduler(
        mode="get_fpc_via_market",
    )
    create_scheduler(
        mode="get_fpc_from_chevent_log",
    )
    create_scheduler(
        mode="get_fpc_via_zen",
    )
    create_scheduler(
        mode="get_fpc_via_serp",
    )
    create_scheduler(
        mode="get_fpc_via_yclid",
    )
    create_scheduler(
        mode="get_fpc_from_watch_log",
    )
    create_scheduler(
        mode="build_fingerprints",
    )
    create_scheduler(
        mode="get_fpc_via_fingerprint",
    )
    create_scheduler(
        mode="get_fpc_via_mc_port",
    )
    create_scheduler(
        mode="get_fpc_from_mts",
    )
    create_scheduler(
        mode="get_fpc_via_extfp",
        crit_time=None,
    )
    create_scheduler(
        mode="hourly",
        export_to_is=True,
    )

    return juggler
